package gocmd

import (
	"io/ioutil"
	"os"
	"os/exec"
	"path"
	"sync"
)

type GoCmd struct {
	mutex   *sync.Mutex
	outputs map[string]string
	tmpDir  string
	debug   bool
}

func NewCmd() (*GoCmd, error) {
	tmpDir, err := ioutil.TempDir("", "")
	if err != nil {
		return nil, err
	}

	return &GoCmd{
		tmpDir:  tmpDir,
		outputs: map[string]string{},
		mutex:   &sync.Mutex{},
	}, nil
}

func (g *GoCmd) Debug() {
	g.debug = true
}

func (g *GoCmd) NoDebug() {
	g.debug = false
}

func (g *GoCmd) Build(importPath string) (string, error) {
	g.mutex.Lock()
	defer g.mutex.Unlock()

	// Check if already built.
	if loc, ok := g.outputs[importPath]; ok {
		return loc, nil
	}

	loc := path.Join(g.tmpDir, path.Base(importPath))
	cmd := exec.Command("go", "build", "-o", loc, importPath)
	if g.debug {
		cmd.Stdout = os.Stdout
		cmd.Stderr = os.Stderr
	}

	err := cmd.Run()
	if err != nil {
		return "", err
	}

	g.outputs[importPath] = loc

	return loc, nil
}
