package main

import (
	"bytes"
	"crypto/rand"
	"crypto/sha256"
	"flag"
	"fmt"
	"io/ioutil"
	"log"
	"os"
	"os/exec"
	"path/filepath"
	"strings"
	"time"

	"github.com/PagerDuty/go-pagerduty"
	"github.com/cloudfoundry/gosigar"
	"golang.org/x/crypto/ssh"
)

const (
	defaultFailServiceKey = "4f0f7384481048ec915b423c60c1a9b5" // Application Validation
	defaultWarnServiceKey = "e054440fda6f416daa39b5c4e9c21d37" // Application Validation Warnings
)

var sendAlerts *bool
var failServiceKey, warnServiceKey string

func main() {
	var privateKeyFlag = flag.String("private-key", "", "Private key used to sign created manifest")
	var manifestFlag = flag.String("create-manifest", "", "Create target checksum manifest file")
	var dirFlag = flag.String("dir", ".", "Directory to checksum")
	var checkAllVersions = flag.Bool("all-versions", false, "Check manifests of all installed versions.")
	flag.StringVar(&failServiceKey, "fail-service-key", defaultFailServiceKey, "Pagerduty service key that will receive alerts.")
	flag.StringVar(&warnServiceKey, "warn-service-key", defaultWarnServiceKey, "Pagerduty service key that will receive warnings.")
	sendAlerts = flag.Bool("alerts", true, "Send alerts to pagerduty")
	interval := flag.Duration("interval", time.Duration(5)*time.Minute, "Interval between every check (default=5m).")
	uptimeWait := flag.Duration("uptime-wait", time.Duration(5)*time.Minute, "Wait after this time after system startup before checking (default=5m).")
	flag.Parse()

	if *privateKeyFlag != "" && *manifestFlag != "" {
		manifest, err := createManifest(*dirFlag)
		if err != nil {
			log.Fatal(err)
		}

		err = ioutil.WriteFile(*manifestFlag, manifest, 0644)
		if err != nil {
			log.Fatal(err)
		}

		err = sign(*privateKeyFlag, *manifestFlag)
		if err != nil {
			log.Fatal(err)
		}
	} else {
		configInfo, err := os.Lstat("/etc/courierd.conf")
		if err != nil {
			fail("", err.Error())
			log.Fatal(err)
		}
		if configInfo.Mode().String() != "-r--r-----" {
			fail("", fmt.Sprintf("/etc/courierd.conf has the wrong file permissions %v", configInfo.Mode().String()))
			log.Fatal(err)
		}
		config, err := ioutil.ReadFile("/etc/courierd.conf")
		if err != nil {
			fail("", err.Error())
			log.Fatal(err)
		}
		log.Printf("Config:\n%v", string(config))
		apps := strings.Split(strings.TrimSpace(string(config)), "\n")
		monitoring := false
		for _, app := range apps {
			app = strings.TrimSpace(app)
			if app != "" && !strings.HasPrefix(app, "#") {
				go monitorApp(app, *checkAllVersions, *interval, *uptimeWait)
				monitoring = true
			}
		}
		if monitoring {
			select {}
		} else {
			for {
				log.Printf("Nothing to monitor.  Sleeping...")
				time.Sleep(*interval)
			}
		}
	}
}

// Entry point for monitoring, either all deployments or the current one.
// Will re-run itself periodically after a given interval.
func monitorApp(app string, checkAllVersions bool, interval time.Duration, uptimeWait time.Duration) {
	uptime := sigar.Uptime{}
	if checkAllVersions {
		log.Printf("Checking all deploys in " + app)
	} else {
		log.Printf("Checking current deploy in " + app)
	}
	for {
		uptime.Get()
		if uptime.Length > uptimeWait.Seconds() {
			deploys, err := ioutil.ReadDir(app + "/releases")
			if err != nil {
				fail(app, err.Error())
			} else if checkAllVersions {
				monitorAppVersions(app, deploys)
			} else {
				monitorAppCurrent(app, deploys)
			}

		} else {
			log.Printf("Server booted %s ago. We start checking after %s. Waiting %s for the next check.", uptime.Format(), uptimeWait, interval)
		}
		time.Sleep(interval)
	}
}

// Verify the checksums of all installed versions of the app.
func monitorAppVersions(app string, deploys_found []os.FileInfo) {
	for _, deploy := range deploys_found {
		checkAppDir(app, deploy)
	}
}

// Verify the checksum of only the current installed version of the app.
func monitorAppCurrent(app string, deploys_found []os.FileInfo) {
	realpath, err := filepath.EvalSymlinks(app + "/current")
	if err != nil {
		warn(app, err.Error())
		return
	}
	for _, deploy := range deploys_found {
		dir := getReleaseDirectory(app, deploy)
		if dir == realpath {
			checkAppDir(app, deploy)
		} else {
			log.Printf("Ignoring `%s` because it's not current.", dir)
		}
	}
}

// Return the release directory of a deployment
func getReleaseDirectory(app string, deploy os.FileInfo) string {
	return app + "/releases/" + deploy.Name()
}

// Return the path for a deployment manifest
func getManifestLocation(app string, deploy os.FileInfo) string {
	return app + "/manifests/" + strings.Split(deploy.Name(), "-")[1] + ".manifest"
}

// Given an application directory, verify the integrity of the files
func checkAppDir(app string, deploy os.FileInfo) {
	manifestLocation := getManifestLocation(app, deploy)
	manifestHashLocation := manifestLocation + ".hash"
	buildManifest, err := ioutil.ReadFile(manifestLocation)
	if err != nil {
		fail(app, err.Error())
		return
	}
	dir := getReleaseDirectory(app, deploy)
	checkManifestHash(&dir, &manifestHashLocation, buildManifest)
	checkManifest(&dir, buildManifest)
}

func checkManifestHash(dir *string, signatureFileLocation *string, buildManifest []byte) {
	signatureBytes, err := ioutil.ReadFile(*signatureFileLocation)
	if err != nil {
		fail(*dir, err.Error())
		return
	}
	log.Printf("Checking signature for " + *dir)
	signature := ssh.Signature{Format: "ssh-rsa", Blob: signatureBytes}
	pubBytes, err := ioutil.ReadFile("/etc/courierd_key.pub")
	if err != nil {
		fail(*dir, err.Error())
		return
	}
	publicKey, _, _, _, err := ssh.ParseAuthorizedKey(pubBytes)
	if err != nil {
		fail(*dir, err.Error())
		return
	}
	err = publicKey.Verify(buildManifest, &signature)
	if err != nil {
		fail(*dir, err.Error())
		return
	}
	log.Printf("Signature matched!")
}

func sign(privateKeyPath string, manifestPath string) error {
	manifest, err := ioutil.ReadFile(manifestPath)
	if err != nil {
		return err
	}
	privBytes, err := ioutil.ReadFile(privateKeyPath)
	if err != nil {
		return err
	}
	privateKey, err := ssh.ParseRawPrivateKey(privBytes)
	if err != nil {
		return err
	}
	signer, err := ssh.NewSignerFromKey(privateKey)
	if err != nil {
		return err
	}
	signature, err := signer.Sign(rand.Reader, manifest)
	if err != nil {
		return err
	}
	err = ioutil.WriteFile(manifestPath+".hash", signature.Blob, 0644)
	if err != nil {
		return err
	}
	return nil
}

func createManifest(dir string) ([]byte, error) {
	var sums []byte

	if !strings.HasSuffix(dir, "/") {
		dir += "/"
	}

	err := filepath.Walk(dir, func(path string, info os.FileInfo, err error) error {
		// NB: filepath.Walk walks in lexiographic order, so the output
		// of this is deterministic and manifests can be compared byte-for-byte
		hash := "-"
		var size int64 = 0
		if info.Mode()&os.ModeSymlink == os.ModeSymlink {
			hash, _ = os.Readlink(path)
		} else if !info.IsDir() {
			contents, _ := ioutil.ReadFile(path)
			hash = fmt.Sprintf("%x", sha256.Sum256(contents))
			size = info.Size()
		}
		filename := strings.Replace(path, dir, "", 1)
		data := fmt.Sprintf("%v, %v, %v, %v\n", filename, size, info.Mode(), hash)
		sums = append(sums, data...)
		return nil
	})
	if err != nil {
		return nil, err
	}

	return sums, nil
}

func checkManifest(dir *string, buildManifest []byte) {
	log.Printf("Checking %v", *dir)
	manifest, err := createManifest(*dir)
	if err != nil {
		fail(*dir, err.Error())
		return
	}
	if bytes.Equal(buildManifest, manifest) {
		log.Printf("Manifests match")
	} else {
		_ = ioutil.WriteFile("/tmp/original_manifest", buildManifest, 0644)
		_ = ioutil.WriteFile("/tmp/deployed_manifest", manifest, 0644)
		cmd := exec.Command("diff", "-u", "/tmp/original_manifest", "/tmp/deployed_manifest")
		out, _ := cmd.CombinedOutput()
		log.Print("DIFF:")
		diffStr := string(out)
		log.Print(diffStr)
		err = sendAlert(failServiceKey, *dir, map[string]string{"message": "Manifest mismatch!", "diff": diffStr})
		if err == nil {
			log.Printf("Alert Sent!")
		}
	}
}

func fail(dir, message string) {
	sendAlert(failServiceKey, dir, map[string]string{"message": message})
}

func warn(dir, message string) {
	sendAlert(warnServiceKey, dir, map[string]string{"message": message})
}

func sendAlert(serviceKey, dir string, details map[string]string) (err error) {
	log.Print(details["message"])
	if *sendAlerts {
		err = createPagerdutyEvent(serviceKey, dir, details)
		if err != nil {
			err = fmt.Errorf("ERROR SENDING ALERT TO PAGERDUTY: %v", err)
		}
	} else {
		err = fmt.Errorf("ALERT NOT SENT!")
	}
	if err != nil {
		log.Print(err)
	}
	return
}

func createPagerdutyEvent(serviceKey, dir string, details map[string]string) (err error) {
	hostname, err := os.Hostname()
	if err != nil {
		log.Fatal(err)
	}
	details["host"] = hostname
	event := pagerduty.Event{
		Type:        "trigger",
		ServiceKey:  serviceKey,
		Description: fmt.Sprintf("Source code checksum mismatch for %v", dir),
		IncidentKey: fmt.Sprintf(`%v%v`, hostname, dir),
		Details:     details,
	}
	_, err = pagerduty.CreateEvent(event)
	return
}
