package execproc

import (
	"flag"
	"fmt"
	"log"
	"os"

	"github.com/mitchellh/cli"
)

type ExecCommand struct {
	Ui        cli.Ui
	debug     bool
	app       string
	env       string
	consulApi string
}

func (vc *ExecCommand) setupFlags() *flag.FlagSet {
	flags := flag.NewFlagSet("exec", flag.ExitOnError)
	flags.StringVar(&vc.app, "app", os.Getenv("APP"), "Application Name for Consul")
	flags.StringVar(&vc.env, "env", os.Getenv("ENVIRONMENT"), "Environment for Consul")
	flags.StringVar(&vc.consulApi, "consul-api", os.Getenv("CONSUL_API"), "Consul API Endpoint")
	flags.BoolVar(&vc.debug, "debug", false, "Enable debug logging")
	return flags
}

func (vc *ExecCommand) Help() string {
	flags := vc.setupFlags()
	flags.Usage()
	return ""
}

func (vc *ExecCommand) Run(args []string) int {
	flags := vc.setupFlags()
	if err := flags.Parse(args); err != nil {
		log.Fatal(err)
	}

	remainingArgs := flags.Args()

	if vc.env == "" {
		log.Fatalf("Must set ENVIRONMENT")
	}

	if vc.app == "" {
		log.Fatalf("Must set APP")
	}

	if vc.consulApi == "" {
		vc.consulApi = "localhost:8500"
	}

	if len(remainingArgs) == 0 {
		fmt.Fprintf(os.Stderr, "Invalid number of arguments\n")
		return 1
	}

	if err := vc.execChild(remainingArgs); err != nil {
		fmt.Fprintf(os.Stderr, "Error running command: %v\n", err)
		return 1
	}

	return 0
}

func (vc *ExecCommand) Synopsis() string {
	return "Exec a command from Procfile"
}
