package execproc

import (
	"fmt"
	"log"
	"os"
	"os/exec"
	"syscall"

	"code.justin.tv/release/libforerunner/pkg/interchange"
)

type prepareCMDFunc func(*exec.Cmd)

var (
	prepCMD prepareCMDFunc
)

func (vc *ExecCommand) execChild(args []string) error {
	interchangeLoc, err := interchange.LoadInterchange(vc.app, vc.env, vc.consulApi)
	if err != nil {
		return fmt.Errorf("Error loading interchange: %v", err)
	}

	command := args[0]
	args = args[1:]

	cmd := exec.Command(command, args...)

	if prepCMD != nil {
		prepCMD(cmd)
	}

	cmd.Env = append(os.Environ(), fmt.Sprintf("%s=%s", interchange.EnvName, interchangeLoc))
	cmd.Stdout = os.Stdout
	cmd.Stderr = os.Stderr

	procCh := make(chan *os.Process, 1)
	go signalWatcher(procCh)
	defer close(procCh)

	if vc.debug {
		vc.Ui.Info(fmt.Sprintf(
			"Forerunner is PID [%d] owned by [%d]",
			os.Getpid(),
			os.Getppid(),
		))
	}
	err = cmd.Start()
	if err != nil {
		return fmt.Errorf("Error starting %q: %v", command, err)
	}
	log.Printf("Started %q", command)

	procCh <- cmd.Process

	err = cmd.Wait()
	// If there wasn't an error we are all good!
	if err == nil {
		return nil
	}

	log.Printf("Child %s exited with error: %v", command, err)

	if exitErr, ok := err.(*exec.ExitError); ok {
		// TODO: Check if this succeeded to prevent windows panic.
		exitCode := exitErr.Sys().(syscall.WaitStatus).ExitStatus()

		vc.Ui.Error(
			fmt.Sprintf(
				"Command %q (Pid: %d) exited with code: %d",
				command,
				exitErr.Pid(),
				exitCode,
			))

		os.Exit(exitCode)
	}

	return err
}
