// +build integration

// The above blank line is required!!!
//
// Integration Tests will only run if you specify the -tags=integration flag to
// go test.
package tests

import (
	"bufio"
	"fmt"
	"log"
	"os"
	"os/exec"
	"path"
	"regexp"
	"strings"
	"syscall"
	"testing"
)

var (
	bin string
)

func init() {
	var err error

	bin, err = builder.Build(ImportPath)
	if err != nil {
		log.Fatal(err)
	}
}

func TestAPPDIR(t *testing.T) {
	curDir, err := os.Getwd()
	if err != nil {
		t.Fatalf("Can't get working directory: %v", err)
	}

	if curDir == "/" {
		t.Fatal("Can't run TestAPPDIR from root.")
	}

	if err := os.Chdir("/"); err != nil {
		t.Fatalf("Can't change dir")
	}
	defer func() {
		if err := os.Chdir(curDir); err != nil {
			t.Fatal("Can't change back to starting directory, all tests are hopeless!")
		}
	}()

	cmd := exec.Command(bin, "exec", "scripts/pwd.sh")
	cmd.Stderr = os.Stderr
	setupEnv(cmd)

	output, err := cmd.Output()
	if err != nil {
		t.Fatalf(err.Error())
	}

	// Strip newlines:
	stdout := strings.TrimSpace(string(output))
	if path.Clean(stdout) != AppDir {
		t.Errorf(`Incorrect response %q: %q; want %q`, "forerunner exec pwd", stdout, AppDir)
	}
}

func TestFlags(t *testing.T) {
	cmd := exec.Command(bin, "exec", "-env", "test", "-app", "forerunner", "scripts/pretty_combined.sh")
	cmd.Env = []string{fmt.Sprintf("APPDIR=%s", AppDir)}
	err := cmd.Run()
	if err != nil {
		t.Fatalf("Error running processed with command line flags: %v", err)
	}
}

func TestLoadConsul(t *testing.T) {
	trace := "shibboleth"

	cmd := exec.Command(bin, "exec", "scripts/pretty_combined.sh")
	setupEnv(cmd)

	output, err := cmd.Output()
	if err != nil {
		t.Fatalf(err.Error())
	}

	if !strings.Contains(string(output), trace) {
		t.Fatalf("Unable to find %q in processed", trace)
	}
}

// TestSignalHandler attempts to test that the child process is being proxied
// signals by forerunner.
func TestSignalHandler(t *testing.T) {
	cmd := exec.Command(bin, "exec", "scripts/signal_catcher.sh")
	setupEnv(cmd)

	stdout, err := cmd.StdoutPipe()
	if err != nil {
		t.Fatal(err.Error())
	}
	defer stdout.Close()
	cmd.Stderr = os.Stderr

	if err := cmd.Start(); err != nil {
		t.Fatalf("Error starting signal_catcher: %v", err)
	}

	scanner := bufio.NewScanner(stdout)
	for scanner.Scan() {
		match, err := regexp.MatchString(`^Ready`, scanner.Text())
		if err != nil {
			t.Fatal(err)
		}

		if match {
			t.Log("Got ready signal from signal_catcher proceeding")
			break
		}
	}
	if err := scanner.Err(); err != nil {
		t.Fatalf("reading standard input:", err)
	}

	t.Logf("Command has exited (should be nil): %v", cmd.ProcessState)
	if err := cmd.Process.Signal(syscall.SIGTERM); err != nil {
		t.Fatalf("Error killing signal_catcher: %v", err)
	}

	t.Logf("Command has exited (should be nil): %v", cmd.ProcessState)
	// Normally a SIGTERM would cause the exited flag to be set to false. Which
	// Go interprets as an error. In this case signal_catcher is designed to
	// exit 0, (non-error) if it gets a SIGTERM.
	if err := cmd.Wait(); err != nil {
		t.Fatalf("Want exit code 0; got: %v", err)
	}
}

// TestArgs tests that args specified in the Procfile.toml and args specified
// on the command line are both combined when forerunner is called.
func TestArgs(t *testing.T) {
	cmd := exec.Command(bin, "exec", "/bin/echo", "a", "b", "c", "d")
	setupEnv(cmd)

	output, err := cmd.Output()
	if err != nil {
		t.Fatal(err)
	}

	cleanOutput := strings.TrimSpace(string(output))
	expectedOutput := "a b c d"
	if expectedOutput != cleanOutput {
		t.Error("echo_args returned %q; want %q", cleanOutput, expectedOutput)
	}
}
