package httpjsonlogger

import (
	"encoding/json"
	"io"
	"net"
	"net/http"
	"net/url"
	"strconv"
	"time"
)

type responseLogger struct {
	w      http.ResponseWriter
	status int
	size   int
}

func (l *responseLogger) Header() http.Header {
	return l.w.Header()
}

func (l *responseLogger) Write(b []byte) (int, error) {
	if l.status == 0 {
		l.status = http.StatusOK
	}
	size, err := l.w.Write(b)
	l.size += size
	return size, err
}

func (l *responseLogger) WriteHeader(s int) {
	l.w.WriteHeader(s)
	l.status = s
}

func (l *responseLogger) Status() int {
	return l.status
}

func (l *responseLogger) Size() int {
	return l.size
}

type loggingResponseWriter interface {
	http.ResponseWriter
	Status() int
	Size() int
}

type hijackLogger struct {
	responseLogger
}

func buildCommonLogLine(req *http.Request, url url.URL, ts time.Time, status int, size int) []byte {
	username := "-"
	if url.User != nil {
		if name := url.User.Username(); name != "" {
			username = name
		}
	}
	host, port, err := net.SplitHostPort(req.RemoteAddr)
	if err != nil {
		host = req.RemoteAddr
	}
	uri := url.RequestURI()
	m := map[string]string{"host": host, "port": port, "username": username, "timestamp": ts.Format("02/Jan/2006:15:04:05 -0700"), "method": req.Method, "uri": uri, "proto": req.Proto, "status": strconv.Itoa(status), "size": strconv.Itoa(size), "referer": req.Referer(), "useragent": req.UserAgent()}
	enc, _ := json.Marshal(m)
	enc = append(enc, "\n"...)
	return enc
}

func writeCombinedLog(w io.Writer, req *http.Request, url url.URL, ts time.Time, status, size int) {
	buf := buildCommonLogLine(req, url, ts, status, size)
	w.Write(buf)
}

type jsonLoggingHandler struct {
	writer  io.Writer
	handler http.Handler
}

func JSONLoggingHandler(out io.Writer, h http.Handler) http.Handler {
	return jsonLoggingHandler{out, h}
}

func (h jsonLoggingHandler) ServeHTTP(w http.ResponseWriter, req *http.Request) {
	t := time.Now()
	var logger loggingResponseWriter
	if _, ok := w.(http.Hijacker); ok {
		logger = &hijackLogger{responseLogger: responseLogger{w: w}}
	} else {
		logger = &responseLogger{w: w}
	}
	url := *req.URL
	h.handler.ServeHTTP(logger, req)
	writeCombinedLog(h.writer, req, url, t, logger.Status(), logger.Size())
}
