package jenkins

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
	"net/url"
	"regexp"
	"strconv"
	"time"
)

type Client struct {
	client  *http.Client
	BaseURL *url.URL
	Auth    *auth
}

type auth struct {
	User     string
	Password string
}

type QueueItem struct {
	Cancelled  bool
	Executable *Build
}

type Build struct {
	Number            int
	URL               string
	FullDisplayName   string
	Duration          int
	EstimatedDuration int
	Timestamp         int64
	Result            string
}

func NewClient(host string, httpClient *http.Client) *Client {
	if httpClient == nil {
		httpClient = http.DefaultClient
	}
	baseURL, _ := url.Parse(host)
	c := &Client{client: httpClient, BaseURL: baseURL}
	return c
}

func (c *Client) SetAuth(user, password string) {
	c.Auth = &auth{User: user, Password: password}
}

func (c *Client) NewRequest(method, urlStr string, body interface{}, query *url.Values) (*http.Request, error) {
	rel, err := url.Parse(urlStr)
	if err != nil {
		return nil, err
	}

	if query != nil {
		rel.RawQuery = query.Encode()
	}

	u := c.BaseURL.ResolveReference(rel)

	buf := new(bytes.Buffer)
	if body != nil {
		buf = bytes.NewBufferString(body.(string))
	}
	req, err := http.NewRequest(method, u.String(), buf)
	if err != nil {
		return nil, err
	}

	if c.Auth != nil {
		req.SetBasicAuth(c.Auth.User, c.Auth.Password)
	}

	return req, nil
}

func (c *Client) Do(req *http.Request, v interface{}) (*http.Response, error) {
	resp, err := c.client.Do(req)
	if err != nil {
		return nil, err
	}

	defer resp.Body.Close()

	if resp.StatusCode > 400 {
		out, err := ioutil.ReadAll(resp.Body)
		if err != nil {
			return resp, err
		}
		return resp, fmt.Errorf("request failed (%v): %v", resp.StatusCode, string(out))
	}

	if v != nil {
		err = json.NewDecoder(resp.Body).Decode(v)
	}

	return resp, err
}

func (c *Client) Build(job string) (int, error) {
	request, err := c.NewRequest("POST", "/job/"+job+"/build", nil, nil)
	if err != nil {
		return 0, fmt.Errorf("error creating request: %v", err)
	}

	return c.runBuildRequest(request)
}

func (c *Client) BuildWithParameters(job string, params map[string]string) (int, error) {
	data := url.Values{}

	for k, v := range params {
		data.Set(k, v)
	}

	request, err := c.NewRequest("POST", "/job/"+job+"/buildWithParameters", data.Encode(), nil)
	if err != nil {
		return 0, fmt.Errorf("error creating request: %v", err)
	}

	request.Header.Add("Content-Type", "application/x-www-form-urlencoded")

	return c.runBuildRequest(request)
}

func (c *Client) runBuildRequest(request *http.Request) (int, error) {
	response, err := c.Do(request, nil)
	if err != nil {
		return 0, fmt.Errorf("error starting job: %v", err)
	}

	queueItemURL, err := response.Location()
	if err != nil {
		return 0, fmt.Errorf("error grabbing location header: %v", err)
	}

	re := regexp.MustCompile("/queue/item/(\\d*)/")
	matches := re.FindSubmatch([]byte(queueItemURL.String()))
	if len(matches) == 0 {
		return 0, fmt.Errorf("error parsing queue item number from location header: %v", queueItemURL.String())
	}

	num, err := strconv.Atoi(string(matches[1]))
	if err != nil {
		return 0, fmt.Errorf("error converting queue number to int: %v", err)
	}

	return num, nil
}

func (c *Client) GetQueueItem(num int) (*QueueItem, error) {
	var queueItem QueueItem

	request, err := c.NewRequest("GET", "/queue/item/"+strconv.Itoa(num)+"/api/json", nil, nil)
	if err != nil {
		return nil, fmt.Errorf("error creating request: %v", err)
	}

	_, err = c.Do(request, &queueItem)
	if err != nil {
		return nil, fmt.Errorf("error getting queue item: %v", err)
	}

	return &queueItem, nil
}

func (c *Client) GetBuild(job string, num int) (*Build, error) {
	var build Build

	request, err := c.NewRequest("GET", "/job/"+job+"/"+strconv.Itoa(num)+"/api/json", nil, nil)
	if err != nil {
		return nil, fmt.Errorf("error creating request: %v", err)
	}

	_, err = c.Do(request, &build)
	if err != nil {
		return nil, fmt.Errorf("error getting build: %v", err)
	}

	return &build, nil
}

const (
	WaitForJobRetry                = 2 // 0 = no retry
	WaitForJobRetryDelayMultiplier = (time.Second)
)

func (c *Client) WaitForJob(job string, queueItemNumber int, tracking chan *Build) (*Build, error) {
	if tracking != nil {
		defer close(tracking)
	}

	var queueItem *QueueItem
	var build *Build
	var err error

	for retry := 0; ; {
		queueItem, err = c.GetQueueItem(queueItemNumber)
		if err != nil {
			if retry < WaitForJobRetry {
				retry++
				time.Sleep(time.Duration(retry) * WaitForJobRetryDelayMultiplier)
				continue
			}
			return nil, err
		}

		if queueItem.Executable != nil {
			break
		}

		time.Sleep(1 * time.Second)
	}

	for retry := 0; ; {
		build, err = c.GetBuild(job, queueItem.Executable.Number)
		if err != nil {
			if retry < WaitForJobRetry {
				retry++
				time.Sleep(time.Duration(retry) * WaitForJobRetryDelayMultiplier)
				continue
			}
			return nil, err
		}

		if tracking != nil {
			tracking <- build
		}

		if build.Result != "" {
			break
		}

		time.Sleep(5 * time.Second)
	}

	if build.Result != "SUCCESS" {
		return build, fmt.Errorf("Job failed with result: %v. Link: %v", build.Result, build.URL)
	}

	return build, nil
}
