package jenkins

import (
	"fmt"
	"net/url"
	"path"
)

// ComputerInfo is a badly (generated) struct that represents the output of
// `http://<jenkins_server>/computer/jenkins2/api/json
type ComputerInfo struct {
	DisplayName         string      `json:"displayName"`
	Executors           []Executor  `json:"executors"`
	Icon                string      `json:"icon"`
	IconClassName       string      `json:"iconClassName"`
	Idle                bool        `json:"idle"`
	JnlpAgent           bool        `json:"jnlpAgent"`
	LaunchSupported     bool        `json:"launchSupported"`
	ManualLaunchAllowed bool        `json:"manualLaunchAllowed"`
	NumExecutors        float64     `json:"numExecutors"`
	Offline             bool        `json:"offline"`
	OfflineCause        interface{} `json:"offlineCause"`
	OfflineCauseReason  string      `json:"offlineCauseReason"`
	OneOffExecutors     []Executor  `json:"oneOffExecutors"`
	TemporarilyOffline  bool        `json:"temporarilyOffline"`
	// MonitorData         *MonitorData `json:"monitorData"`
	// LoadStatistics      map[string]interface{} `json:"loadStatistics"`
	// Actions             []interface{}          `json:"actions"`
}

type Executor struct {
	CurrentExecutable *CurrentExecutable `json:"currentExecutable"`
	Idle              bool               `json:"idle"`
	LikelyStuck       bool               `json:"likelyStuck"`
	Number            float64            `json:"number"`
	Progress          float64            `json:"progress"`
	// CurrentWorkUnit   struct{}           `json:"currentWorkUnit"`
}

type CurrentExecutable struct {
	Number float64 `json:"number"`
	URL    string  `json:"url"`
}

/*
type MonitorData struct {
	Hudson_NodeMonitors_ArchitectureMonitor string `json:"hudson.node_monitors.ArchitectureMonitor"`
	Hudson_NodeMonitors_ClockMonitor        struct {
		Diff float64 `json:"diff"`
	} `json:"hudson.node_monitors.ClockMonitor"`
	Hudson_NodeMonitors_DiskSpaceMonitor struct {
		Path string  `json:"path"`
		Size float64 `json:"size"`
	} `json:"hudson.node_monitors.DiskSpaceMonitor"`
	Hudson_NodeMonitors_ResponseTimeMonitor struct {
		Average float64 `json:"average"`
	} `json:"hudson.node_monitors.ResponseTimeMonitor"`
	Hudson_NodeMonitors_SwapSpaceMonitor struct {
		AvailablePhysicalMemory float64 `json:"availablePhysicalMemory"`
		AvailableSwapSpace      float64 `json:"availableSwapSpace"`
		TotalPhysicalMemory     float64 `json:"totalPhysicalMemory"`
		TotalSwapSpace          float64 `json:"totalSwapSpace"`
	} `json:"hudson.node_monitors.SwapSpaceMonitor"`
	Hudson_NodeMonitors_TemporarySpaceMonitor struct {
		Path string  `json:"path"`
		Size float64 `json:"size"`
	} `json:"hudson.node_monitors.TemporarySpaceMonitor"`
}
*/

func (c *Client) GetComputer(computerName string) (*ComputerInfo, error) {
	query := &url.Values{}

	// We need it to recurse and lookup executor info:
	query.Add("depth", "1")
	request, err := c.NewRequest(
		"POST",
		path.Join("computer", computerName, "api", "json"),
		nil,
		query,
	)
	if err != nil {
		return nil, err
	}

	ci := &ComputerInfo{}
	if _, err := c.Do(request, ci); err != nil {
		return nil, err
	}

	return ci, nil
}

func (c *Client) EnableComputer(computerName string) error {
	ci, err := c.GetComputer(computerName)
	if err != nil {
		return err
	}
	// If already online don't do anything:
	// TODO: This has a race condition.
	if !ci.Offline {
		return nil
	}

	return c.toggleComputer(computerName, "")
}

func (c *Client) DisableComputer(computerName, reason string) error {
	ci, err := c.GetComputer(computerName)
	if err != nil {
		return err
	}
	// If already offline don't do anything:
	// TODO: This has a race condition.
	if ci.Offline {
		return nil
	}

	return c.toggleComputer(computerName, reason)
}

func (c *Client) toggleComputer(computerName, reason string) error {

	query := &url.Values{}

	if reason != "" {
		query.Add("offlineMessage", reason)
	}

	request, err := c.NewRequest(
		"POST",
		path.Join("computer", computerName, "toggleOffline"),
		nil,
		query,
	)
	if err != nil {
		return fmt.Errorf("error creating request: %v", err)
	}

	if _, err := c.Do(request, nil); err != nil {
		return err
	}

	return nil
}
