#!/usr/bin/env python

import codecs
import errno
import logging
import os
import requests
import string
import urllib


# Find all "jenkins.groovy" files in all Twitch repos.  Download them,
# and store them in the right place for Jenkins to use.


OAUTH_TOKEN = os.environ['GITHUB_AUTH']


def mkdir_p(path):
    try:
        os.makedirs(path)
    except OSError as exc:
        if exc.errno == errno.EEXIST and os.path.isdir(path):
            pass
        else:
            raise


def main():
    process_search('filename:jenkins.groovy path:/')


def process_search(q):
    query_string = urllib.urlencode({
        'q': q,
        'per_page': 100
    })
    url = 'https://git.xarth.tv/api/v3/search/code?%s' % query_string
    total_results = 0
    while True:
        logging.info('Getting search results %s', url)
        req = requests.get(url, auth=(OAUTH_TOKEN, ''))

        results = req.json()
        get_page(results)
        total_results += len(results['items'])

        next_link = req.links.get('next') or req.links.get('last')
        if not next_link:
            break

        url = next_link.get('url')
        if not url:
            break
    logging.info('Total results: %d', total_results)


def get_page(results):
    for result in results['items']:
        # The new search query will jenkins.groovy that is not in the root
        # directory. This has an issue here jenkins.groovy in godeps will
        # overwrite the actual jenkins.groovy.
        if result['path'] != 'jenkins.groovy':
            continue

        repo = result['repository']['name']
        org = result['repository']['owner']['login']
        full_name = '%s/%s' % (org, repo)

        if check_if_archived(full_name):
            # This checks to see if the repo is archived and if so skips that repo
            continue

        repo_url = "https://git.xarth.tv/api/v3/repos/%s/%s" % (
            org, repo)
        try:
            repo_payload = requests.get(
                repo_url, auth=(OAUTH_TOKEN, '')).json()
            branch = repo_payload["default_branch"]
        except Exception:
            logging.error('Failed to get default branch for: %s/%s', org, repo)
            continue

        url = "https://git.xarth.tv/raw/%s/%s/%s/%s" % (
            org, repo, branch, result['path'].lstrip('/'))
        logging.info("Downloading %s", url)

        try:
            groovy = requests.get(url, auth=(OAUTH_TOKEN, '')).text
            groovy = groovy.replace('proxy = Jenkins.getInstance().proxy', '')
        except Exception:
            logging.error("Failed to process jenkins.groovy for: %s", url)
            continue

        logging.info('Writing %s/%s...', org, repo)
        mkdir_p('jobs/' + org)

        fname = 'jobs/%s/%s.groovy' % (org, repo.encode('utf-8').translate(string.maketrans('-.', '__')))
        with codecs.open(fname, 'r+', encoding="utf-8") as f:
           if 'git-aws.internal.justin.tv' in f.read():
             logging.error("git-aws.internal.justin.tv is deprecated and should be changed to git.xarth.tv, skipping: %s" % fname)
             continue
           f.seek(0)
           f.write(groovy)

def check_if_archived(repo):
    # This determines if the repo in question is archived and returns a boolean value
    query_string = urllib.urlencode({
        'q': 'archived:true repo:%s' % repo,
    })
    url = 'https://git.xarth.tv/api/v3/search/repositories?%s' % query_string
    req = requests.get(url, auth=(OAUTH_TOKEN, ''))
    results = req.json()
    if results['items'] == []:
        return False
    else:
        logging.info('Repo: %s is archived skipping...', repo)
        return True


if __name__ == "__main__":
    logging.basicConfig(
        format='%(levelname)s %(filename)s:%(lineno)s %(message)s', level=logging.INFO)
    main()
