#!/bin/bash
# This script processes a github webhook event.
# It expects variables 'payload' and 'GITHUB_AUTH' to be set.
# If the payload variable = 'all', this is a special case that will download
# and process ALL jenkins.groovy files it can find in GHE.
# The script will first attempt to process the jenkins.groovy for the repo
# specified in the payload.
# It will then search the list of changed files for templates. If a template is found to be modified
# it will then search Github for all jenkins.groovy files that use the template and process their
# jenkins.groovy files.

set +x # turn off debug. it's ugly and not needed

GITHUB_URL="https://git.xarth.tv"
SYSTEM_TYPE="$(uname -s)"
WORKSPACE="${WORKSPACE:-$(dirname $0)}"
case "$(uname -s)" in
  Linux)
    JQ_URL="https://github.com/stedolan/jq/releases/download/jq-1.5/jq-linux64"
    JQ_MD5="6a342dbb17b2f2ea4ec0e64d2157614d"
    ;;
  Darwin)
    JQ_URL="https://github.com/stedolan/jq/releases/download/jq-1.5/jq-osx-amd64"
    JQ_MD5="81ff0e3ddd999d2f5bd151b882ce7e18"
    ;;
  *)
    echo "Unable to determine your operating system. Exiting"
    exit 1
    ;;
esac
JQ_PATH="/usr/local/bin/jq1.5"
JQ="${JQ_PATH}"
JOB_DIR="${WORKSPACE}/jobs" # location where temporary jenkins.groovy files are stored from repos with jobs that depends on modified templates

function check_exit() {
  # $1: exit status code to check
  # $2: extra message that can be passed
  if [[ $1 -ne 0 ]]; then
    echo "Command returned non-zero exit code: $1" 1>&2
    [[ $2 ]] && echo -e $2 1>&2
    exit $1
  fi
}

function install_jq() {
  [[ -f ${JQ_PATH} ]] && return # return if it's already installed

  echo "Installing jq..."
  wget "${JQ_URL}" -O "${JQ_PATH}"
  md5="$(md5sum ${JQ_PATH})"
  if [[ ! "${md5}" =~ ${JQ_MD5} ]]; then
    rm -f ${JQ_PATH}
    echo "CRITICAL!!"
    echo -e "MD5 sum for jq download did not match.\nexiting\n\n"
    exit 2
  else
    chmod +x ${JQ_PATH}
  fi
}

function set_repo_full_name() {
  fn="$(echo "${payload}" | $JQ -r -e '.repository.full_name')"
  check_exit $? "${payload}\nFailed to determine full name of repository from Github payload"
  REPO_FULL_NAME="${fn}"
}

function set_repo_default_branch() {
  local default_branch="$(echo "${payload}" | $JQ -r -e '.repository.default_branch')"
  check_exit $? "${payload}\nFailed to determine default branch of repository from Github payload"
  REPO_DEFAULT_BRANCH=${default_branch}
}

function search_urls() {
  # Fetches and returns a space separated list of all urls for results in 'SEARCH_URLS_RESULTS' variable
  # $1: search url
  all_urls=""
  url="${1}"
  while [[ "${url}" != "" ]]; do
    [[ -z "${url}" ]] || all_urls="${all_urls} ${url}" # append url if it's not empty
    headers="$(curl -I -s -H "Authorization: token ${GITHUB_AUTH}" "${url}")"
    check_exit $? "Failed to fetch url: ${url}"
    url="$(echo -n "${headers}" | sed -n 's/.*Link: <\(http.*\)>; rel="next".*/\1/p')"
  done
  SEARCH_URLS_RESULTS="${all_urls}"
}

function search_gh() {
  # return via 'SEARCH_GH_RESULTS' variable the results of a search
  # $1: search query
  # $2: optional jq expression to pass through before returning
  # $3: optional extra jq options

  all_json=""
  json=""
  search_urls "${GITHUB_URL}/api/v3/search/${1}"
  if [[ -z "${SEARCH_URLS_RESULTS}" ]]; then
    echo "Something bad happened and no urls were returned. This shouldn't happen" 1>&2
    exit 1
  else
    echo -e "Fetching urls: ${SEARCH_URLS_RESULTS}" 1>&2
  fi
  for this_url in $SEARCH_URLS_RESULTS; do
    (( x++ ))
    json="$(curl -s -H "Authorization: token ${GITHUB_AUTH}" "${this_url}")"
    check_exit $? "Failed to fetch url: ${url}"
    all_json="${all_json} ${json}"
  done
  all_json="$(echo "${all_json}" | $JQ -s '.[]')"
  if [[ -n $2 ]]; then
    SEARCH_GH_RESULTS="$(echo "${all_json}" | $JQ ${3} "${2}")"
  else
    SEARCH_GH_RESULTS="${all_json}"
  fi
}

function get_groovy() {
  # Fetches and writes a jenkins.groovy file to {repo_full_name}.groovy
  # $1: full repo name with jenkins.groovy to fetch
  # $2: optional, defaults to "master", default" branch configured in GitHub for repository

  repo_full_name="${1}"
  repo_default_branch="${2:-master}"
  org="${repo_full_name%%/*}"
  repo_name="${repo_full_name##*/}"
  file="$(curl -s -H "Authorization: token ${GITHUB_AUTH}" "${GITHUB_URL}/raw/${repo_full_name}/${repo_default_branch}/jenkins.groovy")"
  check_exit $? "Failed to fetch url: ${url}"
  if [[ "${file}" != "Not Found" ]]; then
    echo "found jenkins.groovy for repository: ${repo_full_name}"
    mkdir -p $JOB_DIR/$org

    if grep -q 'git-aws.internal.justin.tv' <<< "${file}"; then
      echo "git-aws.internal.justin.tv is deprecated and should be changed to git.xarth.tv, skipping: $repo_full_name"
      exit 1
    fi
    # write to file replacing filenames with '-' with '__'
    # The file also removes occurances of 'proxy = Jenkins.getInstance().proxy'
    echo "${file//proxy = Jenkins.getInstance\(\).proxy/}" > $JOB_DIR/${org}/${repo_name//-/__}.groovy
  else
    echo "No jenkins.groovy found for on ${repo_default_branch} branch in repo: ${repo_full_name}"
    echo "${file}"
  fi
}

function set_template_names() {
  # returns space separated list of template names via TEMPLATE_NAMES variable
  # $1: a list of template files located in this repository
  templates=""
  for this_template in $1; do
    templates="${templates} $(sed -n "s/.*job('\(.*\)').*/\1/p" $this_template | xargs)"
  done
  # xargs is an easy way to clean up multiple spaces and also return empty string if there are only spaces
  TEMPLATE_NAMES="$(echo ${templates} | xargs)"
}

function process_template_deps() {
  # Locates all repos with jenkins.groovy files that depend on any modified template files
  # and then writes them to the current directory with a path based on repo name
  # $1: modified_templates

  # determine template names to search for based on which template files were modified.
  # combine template names with +OR+ to be used in search string
  [[ -z $TEMPLATE_NAMES ]] && set_template_names "$1"
  local template_names="$(echo "${TEMPLATE_NAMES}" | sed 's/ /+OR+/')"
  search_params="in:file+extension:groovy+language:Groovy+filename:jenkins.groovy+path:/&per_page=100"

  # locate all jenkins.groovy files that contain one of the modified templates
  search_gh "code?q=${template_names}+${search_params}" ".items[].repository.full_name" "-r"
  repos="$(echo "${SEARCH_GH_RESULTS}" | xargs)"

  if [[ -n "${repos}" ]]; then
    echo "A template has been modified. Fetching all jenkins.groovy files that depend on it."
    repo_count="$(echo -e "${repos}" | tr ' ' "\n" | wc -l)"
    # write those jenkins.groovy files to local dir for processing.
    for this_repo in ${repos}; do
      echo "Fetching jenkins.groovy from repo: ${this_repo}..."
      get_groovy $this_repo
      echo "$(( --repo_count )) repositories remaining..."
    done
  else
    echo "No repositories found that use the templates: ${template_names}"
  fi
}

if [[ "$(echo -n ${payload} | tr '[:upper:]' '[:lower:]')" == "all" ]]; then
  # This is a special case that we want to build all jobs defined in jenkins.groovy files
  echo "Process ALL jenkins.groovy files"
  ${WORKSPACE}/get-jenkins-jobs.py
  exit $?
fi

# Initialize some important variables
install_jq

# If payload is set, let's parse it to see which repo this event is from.
if [[ -n ${payload} ]]; then
  set_repo_full_name
  set_repo_default_branch
  get_groovy "${REPO_FULL_NAME}" "${REPO_DEFAULT_BRANCH}"
else
  # If there is no payload set, we assume this was triggered by a change to this repo
  # We only want to run this for changes on the master branch
  if [[ "${GIT_BRANCH}" == 'origin/master' ]]; then
#      echo "There was a change to this repo, so we'll fetch all jenkins.groovy files."
#      ${WORKSPACE}/get-jenkins-jobs.py
       echo "Normally, we would run get-jenkins-job.py here to fetch all jenkins.groovy files and process them."
       echo "Unfortunately, that python script is broken at the moment, so we will skip calling it."
  fi
fi
