import javaposse.jobdsl.dsl.helpers.step.*

/**
 * Creates an ElasticBeanstalkVersion for your application
 *
 * ElasticBeanstalk has many ways in which it can be setup. Given the
 * nature of our build system the best pattern is:
 *   - use a Single Container Docker elastic beanstalk setup
 *   - build your code using manta
 *   - copy the salient artifacts into a known location
 *   - use docker to build a docker container
 *   - publish a minimal beanstalk application, referencing your
 * docker container, which can be deployed
 *
 * This function calls a set of sub functions, which you're at liberty
 * to use a la carte if you wish. Specifically
 * buildAndPublishDockerImage is not tied to ElasticBeanstalk. However
 * you'll probably want to just use this wrapper function which
 * bundles everything up and encodes the coupling between the
 * functions.
 *
 * Parameters:
 *  - applicationName should match the name of your ElasticBeanstalk
 * Application
 *  - dockerImage should be a fully qualified, untagged, docker image
 * name, e.g.: docker-registry.dev.us-west2.twitch.tv/test/repo this
 * will be where your docker image is pushed to
 *  - packageDir contains the assets you wish to ultimately deploy
 *  - dockerRunJsonTemplate is described in the comments for
 * createBeanstalkBundle
 *  - ebConfigPath is the path/to/your/.ebextensions
 *  - s3Bucket is the bucket to upload the elasticbeanstalk bundle to
 *
 * Example:
 *   createBeanstalkVersion(
 *     MyTestApplication,
 *     docker-registry.dev.us-west2.twitch.tv/test/repo",
 *     ".manta",
 *     "path/to/Dockerrun.aws.json.template",
 *     "path/to/.ebextensions",
 *     "elasticbeanstalk-us-west-2-123123123123"
 *   )
 */
StepContext.metaClass.createBeanstalkVersion = { String applicationName, String dockerImage, String packageDir, String dockerRunJsonTemplate, String ebConfigPath, String s3Bucket ->
  zipFile = "app.zip"
  sha = "\${GIT_COMMIT}"
  tag = "${dockerImage}:${sha}"
  buildAndPublishDockerImage tag, packageDir
  createBeanstalkBundle dockerRunJsonTemplate, tag, ebConfigPath, zipFile
  uploadBeanstalkBundle applicationName, zipFile, s3Bucket, sha
}

StepContext.metaClass.createBeanstalkVersionRole = { String roleArn, String applicationName, String dockerImage, String packageDir, String dockerRunJsonTemplate, String ebConfigPath, String s3Bucket ->
  zipFile = "app.zip"
  sha = "\${GIT_COMMIT}"
  tag = "${dockerImage}:${sha}"
  buildAndPublishDockerImage tag, packageDir
  createBeanstalkBundle dockerRunJsonTemplate, tag, ebConfigPath, zipFile
  uploadBeanstalkBundleRole roleArn, applicationName, zipFile, s3Bucket, sha
}

/**
 * Builds, tags and publishes the content at packageDir to the
 * specified docker repository.
 *
 * Example:
 *   buildAndPublishDockerImage docker.pkgs.xarth.tv/test/repo:abc123 .manta/
 *
 * See createBeanstalkVersion for an example of this in use
 */
StepContext.metaClass.buildAndPublishDockerImage = { String dockerImageTagged, String packageDir ->
  shell """
docker build -t ${dockerImageTagged} ${packageDir}
docker push ${dockerImageTagged}
"""
}

/**
 * Creates the zip file which can be uploaded to s3 using
 * uploadBeanstalkBundle.
 *
 * It requires, for now, that you pass in Dockerrun.aws.json.template
 * file with the following structure:
 *
 * ```{
 *  "AWSEBDockerrunVersion": "1",
 *  "Image": {
 *    "Name": "<<TAG>>"
 *  },
 *  "Ports": [{
 *    "ContainerPort": "8000"
 *  }],
 *  "Volumes": []
 * }```
 *
 * "<<TAG>>" will be replaced with the value of the dockerImageTagged
 * passed into the function. It is assumed that dockerImagedTagged
 * would be the same value as passed to buildAndPublishDockerImage
 *
 * Ensure to update the ContainerPort value to the primary port for
 * your service.
 *
 * TODO: Replace the dockerrun.aws.json.template with an object
 * instead?
 *
 * Also pass in the path to your .ebextensions directory.
 *
 * Example:
 *   createBeanstalkBundle path/to/Dockerrun.aws.json.template \
 *                         docker.pkgs.xarth.tv/test/repo:abc123 \
 *                         path/to/.ebextensions \
 *                         path/to/zipfile.zip
 *
 * See createBeanstalkVersion for an example of this in use
 */
StepContext.metaClass.createBeanstalkBundle = { String dockerRunJsonTemplate, String dockerImageTagged, String ebConfigPath, String target ->
  shell """
mkdir zipdir
cat ${dockerRunJsonTemplate} | sed \"s#<<TAG>>#${dockerImageTagged}#\" > zipdir/Dockerrun.aws.json
cp -R ${ebConfigPath} zipdir
cd zipdir
zip -r ../${target} * `basename ${ebConfigPath}`
cd ../
"""
}

/**
 * uploads the specified zipfile to the s3bucket provided and calls
 * the elasticbeanstalk API to tag it appropriately and make it
 * available for deployment.
 *
 * This function assumes you are using the credentialsBinding wrapper
 * for your build:
 *
 * ```
 *   wrappers {
 *     credentialsBinding {
 *       string 'AWS_ACCESS_KEY', 'credentials-name-for-your-aws-access-key'
 *       string 'AWS_SECRET_KEY', 'credentials-name-for-your-aws-secret-key'
 *     }
 *   }
 * ```
 *
 * Parameters:
 *  - app the name of the elastic beanstalk application to tag this
 * asset as being applicable to
 *  - filename the path to the zip to upload, recommend creating this
 * zip file using createBeanstalkBundle
 *  - s3bucket the bucket to upload to
 *  - version to tag this asset with, recommend using the gitsha
 *
 * Example:
 *   uploadBeanstalkBundle MyTestApplication \
 *                         path/to/zipfile.zip \
 *                         elasticbeanstalk-us-west-2-465569119046 \
 *                         abc123
 *
 * See createBeanstalkVersion for an example of this in use
 */
StepContext.metaClass.uploadBeanstalkBundle = { String applicationName, String filename, String s3Bucket, String version ->
  shell """
set +x
export AWS_ACCESS_KEY_ID=\$AWS_ACCESS_KEY
export AWS_SECRET_ACCESS_KEY=\$AWS_SECRET_KEY
set -x
export AWS_DEFAULT_REGION=us-west-2
description=`git log --pretty=format:"%an - %ad: %s" --abbrev-commit -1 | cut -c 1-200`
aws s3 cp ${filename} s3://${s3Bucket}/apps/${applicationName}/${version}.zip
aws elasticbeanstalk create-application-version --no-auto-create-application --application-name ${applicationName} --version-label ${version} --description "\$description" --source-bundle S3Bucket=${s3Bucket},S3Key=apps/${applicationName}/${version}.zip
"""
}

StepContext.metaClass.uploadBeanstalkBundleRole = { String roleArn, String applicationName, String filename, String s3Bucket, String version ->
  shell """
set +x
BACKUP_AWS_CONFIG_FILE=\$AWS_CONFIG_FILE
export AWS_CONFIG_FILE=\$(mktemp)
echo -e "[default]\\ncredential_source=Ec2InstanceMetadata\\nrole_arn=${roleArn}" >\$AWS_CONFIG_FILE
set -x
export AWS_DEFAULT_REGION=us-west-2

description=`git log --pretty=format:"%an - %ad: %s" --abbrev-commit -1 | cut -c 1-200`
aws s3 cp ${filename} s3://${s3Bucket}/apps/${applicationName}/${version}.zip
aws elasticbeanstalk create-application-version --no-auto-create-application --application-name ${applicationName} --version-label ${version} --description "\$description" --source-bundle S3Bucket=${s3Bucket},S3Key=apps/${applicationName}/${version}.zip

set +x
rm \$AWS_CONFIG_FILE
export AWS_CONFIG_FILE=\$BACKUP_AWS_CONFIG_FILE
"""
}

/**
 * Zip and upload a Ruby Passenger Artifact for Elastic Beanstalk
 * Use the elasticbeanstalk API to tag it appropriately and make it
 * available for deployment.
 *
 * This function assumes you are using the credentialsBinding wrapper
 * for your build:
 *
 * ```
 *   wrappers {
 *     credentialsBinding {
 *       string 'AWS_ACCESS_KEY', 'credentials-name-for-your-aws-access-key'
 *       string 'AWS_SECRET_KEY', 'credentials-name-for-your-aws-secret-key'
 *     }
 *   }
 * ```
 *
 * Parameters:
 *  - applicationName - the identifier of the Beanstalk Application to install the artifact to
 *  - version - filename for the created artifact
 *  - s3bucket the bucket to upload to
 *
 * Example:
 *   uploadBeanstalkBundle MyTestApplication \
 *                         abc123-master \
 *                         elasticbeanstalk-us-west-2-465569119046
 *
 *
 */
StepContext.metaClass.createPassengerArtifact = { String applicationName, String version, String s3Bucket ->
  shell """
set +x
export AWS_ACCESS_KEY_ID=\$AWS_ACCESS_KEY
export AWS_SECRET_ACCESS_KEY=\$AWS_SECRET_KEY
set -x
export AWS_DEFAULT_REGION=us-west-2

zip ./${version}.zip -r ./

description=`git log --pretty=format:"%an - %ad: %s" --abbrev-commit -1 | cut -c 1-200`
aws s3 cp ${version}.zip s3://${s3Bucket}/apps/${applicationName}/${version}.zip
aws elasticbeanstalk create-application-version --no-auto-create-application --application-name ${applicationName} --version-label ${version} --description "\$description" --source-bundle S3Bucket=${s3Bucket},S3Key=apps/${applicationName}/${version}.zip
"""
}

StepContext.metaClass.createPassengerArtifactRole = { String roleArn, String applicationName, String version, String s3Bucket ->
  shell """
set +x
BACKUP_AWS_CONFIG_FILE=\$AWS_CONFIG_FILE
export AWS_CONFIG_FILE=\$(mktemp)
echo -e "[default]\\ncredential_source=Ec2InstanceMetadata\\nrole_arn=${roleArn}" >\$AWS_CONFIG_FILE
set -x
export AWS_DEFAULT_REGION=us-west-2

zip ./${version}.zip -r ./

description=`git log --pretty=format:"%an - %ad: %s" --abbrev-commit -1 | cut -c 1-200`
aws s3 cp ${version}.zip s3://${s3Bucket}/apps/${applicationName}/${version}.zip
aws elasticbeanstalk create-application-version --no-auto-create-application --application-name ${applicationName} --version-label ${version} --description "\$description" --source-bundle S3Bucket=${s3Bucket},S3Key=apps/${applicationName}/${version}.zip

set +x
rm \$AWS_CONFIG_FILE
export AWS_CONFIG_FILE=\$BACKUP_AWS_CONFIG_FILE
"""
}

/**
 * Deploy an artifact to an Elastic Beanstalk application
 *
 * This function assumes you are using the credentialsBinding wrapper
 * for your build:
 *
 * ```
 *   wrappers {
 *     credentialsBinding {
 *       string 'AWS_ACCESS_KEY', 'credentials-name-for-your-aws-access-key'
 *       string 'AWS_SECRET_KEY', 'credentials-name-for-your-aws-secret-key'
 *     }
 *   }
 * ```
 *
 * Parameters:
 *  - applicationName - the identifier of the Beanstalk Application to install the artifact to
 *  - env - the environment to deploy to
 *  - artifactName - the deployable filename
 *
 * Example:
 *   deployElasticBeanstalkArtifact MyTestApplication \
 *                         stage \
 *                         abc123.zip
 *
 * Deploys abc123.zip to MyTestApplication-stage
 *
 */
StepContext.metaClass.deployElasticBeanstalkArtifact = { String applicationName, String env, String artifactName ->
  shell """
set +x
export AWS_ACCESS_KEY_ID=\$AWS_ACCESS_KEY
export AWS_SECRET_ACCESS_KEY=\$AWS_SECRET_KEY
set -x
export AWS_DEFAULT_REGION=us-west-2

mkdir -p .elasticbeanstalk
cat > .elasticbeanstalk/config.yml <<EOF
global:
  application_name: ${applicationName}
  default_region: us-west-2
option_settings:
 - namespace: aws:elasticbeanstalk:command
   option_name: Timeout
   value: 1800
EOF

eb deploy ${applicationName}-${env} --version ${artifactName}
rm .elasticbeanstalk/config.yml
"""
}

StepContext.metaClass.deployElasticBeanstalkArtifactRole = { String roleArn, String applicationName, String env, String artifactName ->
  shell """
set +x
BACKUP_AWS_CONFIG_FILE=\$AWS_CONFIG_FILE
export AWS_CONFIG_FILE=\$(mktemp)
echo -e "[default]\\ncredential_source=Ec2InstanceMetadata\\nrole_arn=${roleArn}" >\$AWS_CONFIG_FILE
set -x
export AWS_DEFAULT_REGION=us-west-2

mkdir -p .elasticbeanstalk
cat > .elasticbeanstalk/config.yml <<EOF
global:
  application_name: ${applicationName}
  default_region: us-west-2
option_settings:
 - namespace: aws:elasticbeanstalk:command
   option_name: Timeout
   value: 1800
EOF

eb deploy ${applicationName}-${env} --version ${artifactName}
rm .elasticbeanstalk/config.yml

set +x
rm \$AWS_CONFIG_FILE
export AWS_CONFIG_FILE=\$BACKUP_AWS_CONFIG_FILE
"""
}

/**
 * Cleanup Builds, there's a 500 artifact limit
 *
 * This function assumes you are using the credentialsBinding wrapper
 * for your build:
 *
 * ```
 *   wrappers {
 *     credentialsBinding {
 *       string 'AWS_ACCESS_KEY', 'credentials-name-for-your-aws-access-key'
 *       string 'AWS_SECRET_KEY', 'credentials-name-for-your-aws-secret-key'
 *     }
 *   }
 * ```
 *
 * Parameters:
 *  - applicationName - the identifier of the Beanstalk Application to install the artifact to
 *
 */
StepContext.metaClass.cleanBeanstalkArtifacts = { String applicationName, int versionsToKeep = 50 ->
  shell """
set +x
export AWS_ACCESS_KEY_ID=\$AWS_ACCESS_KEY
export AWS_SECRET_ACCESS_KEY=\$AWS_SECRET_KEY
set -x
export AWS_DEFAULT_REGION=us-west-2

LIMIT=${versionsToKeep}
VERSIONS="\$(aws elasticbeanstalk describe-application-versions --application-name ${applicationName})"
VERSION_COUNT="\$(echo \${VERSIONS} | jq -r '.ApplicationVersions[] | .VersionLabel' | wc -l)"

if [ \${VERSION_COUNT} -gt \${LIMIT} ]; then
        DELETE_VERSION="\$(echo \${VERSIONS} | jq -r '.ApplicationVersions[] | .VersionLabel' | sed -n \${LIMIT}p)"
        if [ -n "\${DELETE_VERSION}" ]; then
                aws elasticbeanstalk delete-application-version \
                        --application-name ${applicationName} \
                        --version-label \${DELETE_VERSION}
        fi
fi

"""
}

StepContext.metaClass.cleanBeanstalkArtifactsRole = { String roleArn, String applicationName, int versionsToKeep = 50 ->
  shell """
set +x
BACKUP_AWS_CONFIG_FILE=\$AWS_CONFIG_FILE
export AWS_CONFIG_FILE=\$(mktemp)
echo -e "[default]\\ncredential_source=Ec2InstanceMetadata\\nrole_arn=${roleArn}" >\$AWS_CONFIG_FILE
set -x
export AWS_DEFAULT_REGION=us-west-2

LIMIT=${versionsToKeep}
VERSIONS="\$(aws elasticbeanstalk describe-application-versions --application-name ${applicationName})"
VERSION_COUNT="\$(echo \${VERSIONS} | jq -r '.ApplicationVersions[] | .VersionLabel' | wc -l)"
if [ \${VERSION_COUNT} -gt \${LIMIT} ]; then
        DELETE_VERSION="\$(echo \${VERSIONS} | jq -r '.ApplicationVersions[] | .VersionLabel' | sed -n \${LIMIT}p)"
        if [ -n "\${DELETE_VERSION}" ]; then
                aws elasticbeanstalk delete-application-version \
                        --application-name ${applicationName} \
                        --version-label \${DELETE_VERSION}
        fi
fi

set +x
rm \$AWS_CONFIG_FILE
export AWS_CONFIG_FILE=\$BACKUP_AWS_CONFIG_FILE
"""
}

/**
 * Zip and upload an Artifact for Elastic Beanstalk
 * Use the elasticbeanstalk API to tag it appropriately and make it
 * available for deployment.
 *
 * This function assumes you are using the credentialsBinding wrapper
 * for your build:
 *
 * ```
 *   wrappers {
 *     credentialsBinding {
 *       string 'AWS_ACCESS_KEY', 'credentials-name-for-your-aws-access-key'
 *       string 'AWS_SECRET_KEY', 'credentials-name-for-your-aws-secret-key'
 *     }
 *   }
 * ```
 *
 * Parameters:
 *  - applicationName - the identifier of the Beanstalk Application to install the artifact to
 *  - version - filename for the created artifact
 *  - s3bucket the bucket to upload to
 *  - archiveDirectory - relative path of the directory to be archived
 *
 * Example:
 *   createBeanstalkArtifact MyTestApplication \
 *                           abc123-master \
 *                           elasticbeanstalk-us-west-2-465569119046 \
 *                           .manta
 *
 *
 */
StepContext.metaClass.createBeanstalkArtifactFromDirectory = { String applicationName, String version, String s3Bucket, String archiveDirectory = "./" ->
  shell """
set +x
export AWS_ACCESS_KEY_ID=\$AWS_ACCESS_KEY
export AWS_SECRET_ACCESS_KEY=\$AWS_SECRET_KEY
set -x
export AWS_DEFAULT_REGION=us-west-2

cd ${archiveDirectory}
zip -r ./${version}.zip ./

description=`git log --pretty=format:"%an - %ad: %s" --abbrev-commit -1 | cut -c 1-200`
aws s3 cp ${version}.zip s3://${s3Bucket}/apps/${applicationName}/${version}.zip
aws elasticbeanstalk create-application-version --no-auto-create-application --application-name ${applicationName} --version-label ${version} --description "\$description" --source-bundle S3Bucket=${s3Bucket},S3Key=apps/${applicationName}/${version}.zip
"""
}

StepContext.metaClass.createBeanstalkArtifactFromDirectoryRole = { String roleArn, String applicationName, String version, String s3Bucket, String archiveDirectory = "./" ->
  shell """
set +x
BACKUP_AWS_CONFIG_FILE=\$AWS_CONFIG_FILE
export AWS_CONFIG_FILE=\$(mktemp)
echo -e "[default]\\ncredential_source=Ec2InstanceMetadata\\nrole_arn=${roleArn}" >\$AWS_CONFIG_FILE
set -x
export AWS_DEFAULT_REGION=us-west-2

cd ${archiveDirectory}
zip -r ./${version}.zip ./

description=`git log --pretty=format:"%an - %ad: %s" --abbrev-commit -1 | cut -c 1-200`
aws s3 cp ${version}.zip s3://${s3Bucket}/apps/${applicationName}/${version}.zip
aws elasticbeanstalk create-application-version --no-auto-create-application --application-name ${applicationName} --version-label ${version} --description "\$description" --source-bundle S3Bucket=${s3Bucket},S3Key=apps/${applicationName}/${version}.zip

set +x
rm \$AWS_CONFIG_FILE
export AWS_CONFIG_FILE=\$BACKUP_AWS_CONFIG_FILE
"""
}

StepContext.metaClass.uploadBeanstalkApp = { String app, String filename ->
	shell """
export AWS_ACCESS_KEY_ID=\$AWS_ACCESS_KEY
export AWS_SECRET_ACCESS_KEY=\$AWS_SECRET_KEY
aws s3 cp ${filename} s3://elastic-beanstalk-docker/${app}/\$GIT_COMMIT.zip
aws elasticbeanstalk create-application-version --no-auto-create-application --application-name ${app} --version-label \$GIT_COMMIT --source-bundle S3Bucket=elastic-beanstalk-docker,S3Key=${app}/\$GIT_COMMIT.zip
"""
}

StepContext.metaClass.uploadBeanstalkAppRole = { String roleArn, String app, String filename ->
	shell """
set +x
BACKUP_AWS_CONFIG_FILE=\$AWS_CONFIG_FILE
export AWS_CONFIG_FILE=\$(mktemp)
echo -e "[default]\\ncredential_source=Ec2InstanceMetadata\\nrole_arn=${roleArn}" >\$AWS_CONFIG_FILE
set -x
export AWS_DEFAULT_REGION=us-west-2

aws s3 cp ${filename} s3://elastic-beanstalk-docker/${app}/\$GIT_COMMIT.zip
aws elasticbeanstalk create-application-version --no-auto-create-application --application-name ${app} --version-label \$GIT_COMMIT --source-bundle S3Bucket=elastic-beanstalk-docker,S3Key=${app}/\$GIT_COMMIT.zip

set +x
rm \$AWS_CONFIG_FILE
export AWS_CONFIG_FILE=\$BACKUP_AWS_CONFIG_FILE
"""
}
