import javaposse.jobdsl.dsl.Context
import javaposse.jobdsl.dsl.ContextHelper
import javaposse.jobdsl.dsl.Job
import javaposse.jobdsl.dsl.WithXmlAction
import javaposse.jobdsl.dsl.helpers.ScmContext
import javaposse.jobdsl.dsl.helpers.scm.GitContext
import javaposse.jobdsl.dsl.helpers.triggers.TriggerContext


/**
* Adds job-dsl for twitch-github-jenkins plugin.
*
* This uses a bunch of monkey patching in order to provide two bits of job-dsl
* in order to use non-polling build triggering and build selection from
* GitHub Push webhooks.
*
* In Jenkins GitSCM, there are two places where polling is implemented. It's
* implemented in the triggering of builds, to figure out whether it should
* make some builds or not. It's also implemented in "build choosing" to figure
* out what version it will build when it does build.
*
* To use the non-polling code for configuring the trigger, use this:
*   triggers {
*       githubPush {
*           githubPolling(false)
*       }
*   }
*
* To use the non-polling code for selecting the commit to build at, use this:
*   scm {
*       git {
*           remote { ... }
*           githubPolling(false)
*           ...
*       }
*   }
*/

// Save the original implementation of git as an 'originalGit' method.
if (!ScmContext.metaClass.getMetaMethod('originalGit', [Closure] as Class[])) {
  def originalGit = ScmContext.metaClass.getMetaMethod('git', [Closure] as Class[])
  ScmContext.metaClass.originalGit = { Closure gitClosure ->
    originalGit.invoke(delegate, gitClosure)
  }
}

ScmContext.metaClass.git = { Closure gitClosure ->
  // This piece of closure here runs "after" job-dsl git closure in the job-dsl.
  def newGitClosure = gitClosure >> {
    if (!githubPolling) {
      extensions << NodeBuilder.newInstance().'hudson.plugins.git.extensions.impl.BuildChooserSetting' {
        buildChooser(class: 'tv.justin.jenkins.twitchgithub.GitHubBuildChooser')
      }
    }
  }
  delegate.originalGit(newGitClosure)
}


// Add githubPolling property to GitContext objects.
GitContext.metaClass.githubPolling = true
GitContext.metaClass.githubPolling = { boolean enable = true ->
  delegate.githubPolling = enable
}

TriggerContext.metaClass.githubPush = { Closure contextClosure = null ->
  GithubPushContext githubPushContext = new GithubPushContext()
  ContextHelper.executeInContext(contextClosure, githubPushContext)

  if (githubPushContext.githubPolling) {
    triggerNodes << new NodeBuilder().'com.cloudbees.jenkins.GitHubPushTrigger' {
      spec ''
    }
  }
  else {
    triggerNodes << new NodeBuilder().'tv.justin.jenkins.twitchgithub.GitHubPushTrigger' {
      spec ''
    }
  }
}


Job.metaClass.triggers = { Closure contextClosure ->
  TriggerContext context = new TriggerContext(jobManagement, delegate)
  ContextHelper.executeInContext(contextClosure, context)

  def triggerClasses = [
  'com.cloudbees.jenkins.GitHubPushTrigger',
  'tv.justin.jenkins.twitchgithub.GitHubPushTrigger']

  // First delete any already existing GitHubPushTrigger
  withXmlActions << WithXmlAction.create { Node project ->
    if (context.triggerNodes.any {it.name() in triggerClasses}) {
      Node triggers = project / 'triggers'
      triggerClasses.each { triggerClass ->
        triggers.remove(triggers / triggerClass)
      }
    }
    context.triggerNodes.each {
      project / 'triggers' << it
    }
  }
}

class GithubPushContext implements Context {
  boolean githubPolling = true

  void githubPolling(boolean enable = true) {
    githubPolling = enable
  }
}
