package main

import (
	"flag"
	"fmt"
	"log"
	"math/rand"
	"os"
	"regexp"
	"strings"
	"sync"
	"time"

	"github.com/hashicorp/consul/api"
)

func init() {
	rand.Seed(time.Now().UnixNano())
}

var client *api.Client

type QueryOptions struct {
	IncludeDead  bool
	LookupByIP   bool
	serversRegex *regexp.Regexp
}

type Host struct {
	node string
	dc   string
}

func main() {
	var err error
	var xargs, random bool

	var opts QueryOptions

	dc := "all"
	delim := ","
	consulServer := ""

	flag.StringVar(&dc, "dc", dc, "dc to search, default=all")
	flag.StringVar(&delim, "delim", delim, "delimiter to join hosts with")
	flag.StringVar(&consulServer, "api", os.Getenv("CONSUL_HOST"), "consul api server to talk to")
	flag.BoolVar(&xargs, "xargs", xargs, "Print in xargs mode")
	flag.BoolVar(&xargs, "x", xargs, "Print in xargs mode")
	flag.BoolVar(&random, "r", random, "Randomize order of hosts")
	flag.BoolVar(&opts.IncludeDead, "dead", opts.IncludeDead, "Include dead servers")
	flag.BoolVar(&opts.LookupByIP, "ip", opts.LookupByIP, "return ips instead of hostnames because why use a registry to lookup in a registry?")
	flag.Parse()

	if consulServer == "" {
		consulServer = "localhost:8500"
	}

	if xargs {
		delim = "\n"
	}

	regexps := flag.Args()
	if len(regexps) != 0 && len(regexps) != 1 {
		log.Fatalln("Too many regexps defined")
	}

	if len(regexps) > 0 {
		opts.serversRegex, err = compileRegexp(regexps[0])
		if err != nil {
			log.Fatal(err)
		}
	}

	config := api.DefaultConfig()
	config.Address = consulServer
	client, err = api.NewClient(config)
	if err != nil {
		log.Fatal(err)
	}

	servers, err := getServers(consulServer, dc, opts)
	if err != nil {
		log.Fatal(err)
	}

	if random {
		servers = shuffleServers(servers)
	}

	fmt.Println(strings.Join(servers, delim))
}

func compileRegexp(rawRegexp string) (*regexp.Regexp, error) {
	// Maintain compatibility with old liveservers
	if !strings.HasPrefix(rawRegexp, "^") {
		rawRegexp = "^" + rawRegexp
	}
	return regexp.Compile(rawRegexp)
}

func filterWithRegexp(servers []string, re *regexp.Regexp) ([]string, error) {
	match := []string{}
	for _, server := range servers {
		if re.MatchString(server) {
			match = append(match, server)
		}
	}

	return match, nil
}

func getServers(consulServer, dc string, opts QueryOptions) ([]string, error) {
	var err error
	dcs := []string{dc}
	if dc == "all" {
		dcs, err = client.Catalog().Datacenters()
		if err != nil {
			log.Fatal(err)
		}
	}

	wg := &sync.WaitGroup{}
	wg.Add(len(dcs))

	servers := []string{}
	resChan := make(chan []string, len(dcs))

	for _, dc := range dcs {
		go getServersDC(client, dc, opts, resChan, wg)
	}

	wg.Wait()
	close(resChan)

	for response := range resChan {
		servers = append(servers, response...)
	}

	if opts.serversRegex != nil {
		servers, err = filterWithRegexp(servers, opts.serversRegex)
		if err != nil {
			log.Fatal(err)
		}
	}

	if opts.LookupByIP {
		wg := &sync.WaitGroup{}
		wg.Add(len(servers))

		ips := []string{}
		resChan := make(chan string, len(servers))

		for _, n := range servers {

			h := parseHost(n)
			go getNodesDC(client, h, resChan, wg)
		}
		wg.Wait()
		close(resChan)

		for r := range resChan {
			ips = append(ips, r)
		}

		return ips, nil
	}

	return servers, nil
}

func parseHost(fqdn string) Host {
	parts := strings.Split(fqdn, ".")
	if parts[len(parts)-2] != "justin" {
		log.Fatalf("unrecognized FQDN format: %v", parts)
	}
	if parts[len(parts)-1] != "tv" {
		log.Fatalf("unrecognized FQDN format: %v", parts)
	}

	return Host{
		node: strings.Join(parts[0:2], "."),
		dc:   parts[2],
	}
}

func getNodesDC(client *api.Client, h Host, res chan<- string, wg *sync.WaitGroup) {
	defer wg.Done()

	node, _, err := client.Catalog().Node(h.node, &api.QueryOptions{
		Datacenter: h.dc,
	})
	if err != nil {
		log.Printf("Got error fetching %s from %q, %v", h.node, h.dc, err)
		return
	}

	res <- node.Node.Address
}

func getServersDC(client *api.Client, dc string, opts QueryOptions, res chan<- []string, wg *sync.WaitGroup) {
	defer wg.Done()

	servers, err := GetServers(client, dc, opts)
	if err != nil {
		log.Printf("Got error fetching servers from %q: %v", dc, err)
		return
	}

	res <- servers
}

func GetServers(client *api.Client, dc string, opts QueryOptions) ([]string, error) {
	services, _, err := client.Health().Service(
		"nodeinfo",
		"",
		!opts.IncludeDead,
		&api.QueryOptions{
			Datacenter: dc,
		},
	)
	if err != nil {
		return nil, err
	}

	identifiers := []string{}
	for _, service := range services {
		if hostname := parseFQDN(service.Service); hostname != "" {
			identifiers = append(identifiers, hostname)
		}
	}
	return identifiers, nil
}
