package main

import (
	"encoding/json"
	"errors"
	"fmt"
	"net"
	"net/http"
	"net/smtp"
	"os"
	"path"
	"strconv"
	"strings"

	log "github.com/Sirupsen/logrus"
	"github.com/stvp/rollbar"
	"golang.org/x/net/context"

	"code.justin.tv/common/chitin"
	"code.justin.tv/common/golibs/statsd"
	"code.justin.tv/release/libforerunner"
	"code.justin.tv/release/twitch-consul-api"
	"github.com/afex/hystrix-go/hystrix"
	"github.com/gorilla/mux"
	consulapi "github.com/hashicorp/consul/api"
)

var (
	smtpClient   *smtp.Client
	consulClient *consulapi.Client
	statsdClient statsd.Stats
	GitCommit    = "development"
	fr           *libforerunner.Forerunner
)

type Config struct {
	ServiceName       string `description:"The name of the service"`
	ConsulHost        string `description:"Consul hostname"`
	ConsulPrefix      string `description:"Prefix for consul data storage"`
	Port              int    `description:"Port to bind to for HTTP API"`
	StatsdHost        string `description:"Host to send statsd data to"`
	StatsdPort        int    `description:"Port to send statsd data to"`
	RollbarToken      string `forerunner:",secret" description:"Token for talking to rollbar"`
	Environment       string `description:"Environment maelstrom is running in"`
	AmazonSQSURL      string `description:"URL for Amazon SQS Queue"`
	AmazonSQSResource string `description:"Resource Name for Amazon SQS Queue"`
	AmazonAccessKey   string `description:"Amazon Access Key"`
	AmazonSecretKey   string `description:"Amazon Secret Key"`
	AmazonRegion      string `description:"Amazon Region"`
}

var c *Config

func init() {
	var err error

	c = &Config{
		ServiceName:       "maelstrom-worker",
		ConsulHost:        "consul.internal.justin.tv",
		ConsulPrefix:      "application-data/maelstrom-worker/dev",
		Port:              8080,
		StatsdHost:        "graphite.internal.justin.tv",
		StatsdPort:        8135,
		RollbarToken:      "9c22637d84314a28b3d4afe9cf92ea85",
		Environment:       "development",
		AmazonSQSURL:      "https://sqs.us-east-1.amazonaws.com/673385534282/maelstrom-dev",
		AmazonSQSResource: "arn:aws:sqs:us-east-1:673385534282:maelstrom-dev",
		AmazonAccessKey:   "AAAAAAAAAAAAAAAAAAAA",
		AmazonSecretKey:   "AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA",
		AmazonRegion:      "us-west2",
	}

	log.SetFormatter(&log.JSONFormatter{})

	fr, err = libforerunner.Init(&libforerunner.Options{
		DefaultConfig: c,
	})
	if err != nil {
		log.Fatal(err)
	}

	err = fr.GetConfig(c)
	if err != nil {
		log.Fatal(err)
	}

	fmt.Fprintf(os.Stderr, "Running commit %q with config:\n", GitCommit)
	fr.Print()

	configureConsulClient(c)
	configureStatsd(c)
	configureRollbar(c)
}

func configureConsulClient(c *Config) error {
	var err error
	consulClient, err = twitchconsulapi.NewClient(c.ConsulHost, chitin.Client(context.Background()))
	if err != nil {
		log.Fatalf("configureConsulClient: error creating consulapi client: %v", err)
		return errors.New(fmt.Sprintf("configureConsulClient: error creating consulapi client: %v", err))
	}
	c.ConsulPrefix = strings.TrimLeft(path.Clean(c.ConsulPrefix), "/")
	return nil
}

func configureRollbar(c *Config) error {
	rollbar.Token = c.RollbarToken
	rollbar.Environment = c.Environment
	rollbar.CodeVersion = GitCommit
	return nil
}

func configureStatsd(c *Config) error {
	var err error

	if !(len(c.StatsdHost) > 0) {
		statsdClient = statsd.Noop()
	}

	statsdAddr := net.JoinHostPort(c.StatsdHost, strconv.Itoa(c.StatsdPort))
	statsdClient, err = statsd.Dial("udp", statsdAddr, statsd.StatsConfig{
		Rate:   1.0,
		Prefix: c.ServiceName + "." + c.Environment,
	})
	if err != nil {
		log.Fatalf("configureStatsd: error=%q\n", err)
		return errors.New(fmt.Sprintf("configureStatsd: error=%q\n", err))
	}
	log.Printf("configureStatsd connected server=%s\n", statsdAddr)
	return nil
}

func healthCheck(w http.ResponseWriter, req *http.Request) {
	type hcStruct struct {
		GitCommit string
	}
	if err := json.NewEncoder(w).Encode(&hcStruct{
		GitCommit: GitCommit,
	}); err != nil {
		log.Printf("Error encoding healthCheck: %v", err)
	}
}

func main() {
	router := mux.NewRouter()
	router.HandleFunc("/healthcheck", healthCheck)

	hystrixStreamHandler := hystrix.NewStreamHandler()
	hystrixStreamHandler.Start()
	go http.ListenAndServe(net.JoinHostPort("", "7000"), hystrixStreamHandler)

	// TODO setup queue listener and job handlers
}

func mustTraceContext() context.Context {
	ctx, err := chitin.ExperimentalTraceContext(context.Background())
	if err != nil {
		log.Fatal(err)
	}
	return ctx
}
