package jobs

import (
	"encoding/json"
	"errors"
	"fmt"
	"time"

	"code.justin.tv/common/messagequeue"
	log "github.com/Sirupsen/logrus"
	"github.com/aws/aws-sdk-go/aws"
)

// TODO: maelstrom and workers should use the same definition for this:
type Job struct {
	ID                *int64     `json:"id,omitempty" db:"id"`
	UUID              *string    `json:"id,omitempty" db:"uuid"`
	Repository        *string    `json:"repository,omitempty" db:"repository"`
	SHA               *string    `json:"sha,omitempty" db:"sha"`
	Rollback_Strategy *string    `json:"rollback_strategy,omitempty" db:"rollbackstrategy"`
	Discovery         *string    `json:"rollback_strategy,omitempty" db:"discovery"`
	Deployment_Style  *string    `json:"deployment_style,omitempty" db:"deploymentstyle"`
	Environment       *string    `json:"environment,omitempty" db:"environment"`
	Creator           *string    `json:"creator,omitempty" db:"creator"`
	Status            *string    `json:"state,omitempty" db:"status"`
	CreatedAt         *time.Time `json:"created_at,omitempty" db:"createdat"`
	UpdatedAt         *time.Time `json:"updated_at,omitempty" db:"updatedat"`
}

type Queue struct {
	Q *messagequeue.SQSQueue
	R *messagequeue.Receiver
}

type queueLogger struct {
	logger *log.Logger
}

func NewQueueLogger() aws.Logger {
	return &queueLogger{logger: log.New()}
}

func (l queueLogger) Log(args ...interface{}) {
	l.logger.Println(args...)
}

// initialize a new queue connection
func NewQueue(accesskey string, secretkey string, region_name string, resource string) (*Queue, error) {
	c := aws.NewConfig()
	c = c.WithRegion(region_name)
	c = c.WithEndpoint("")
	c = c.WithLogger(NewQueueLogger())
	sqs, err := messagequeue.NewSQSQueue(c, resource)
	if err != nil {
		log.Warn("queue NewQueue: Unable to create new SQS Queue: %v", err)
		return nil, errors.New(fmt.Sprintf("queue NewQueue: Unable to create new SQS Queue: %v", err))
	}
	buffparams := messagequeue.BufferParameters{1, 10 * time.Second}
	r := sqs.Receiver(buffparams)
	q := &Queue{Q: sqs, R: &r}
	return q, nil
}

// grab a job from the queue to work on by a free worker
func (q *Queue) Dequeue() (*Job, error) {
	r := *q.R
	messages, err := r.Receive()
	if err != nil {
		return nil, errors.New(fmt.Sprintf("queue Dequeue: Receive message failed: %v", err))
	}
	log.Debug("queue Dequeue: Received messages: %v", messages)
	msg := messages[0]
	data, err := msg.Bytes()
	var j Job
	err = json.Unmarshal(data, &j)
	if err != nil {
		log.Warn("queue Dequeue: Parsing of message failed: %v", err)
		return nil, errors.New(fmt.Sprintf("queue Dequeue: Parsing of message failed: %v", err))
	}

	log.Debug("queue Dequeue: Received job: %v", j)
	return &j, nil
}
