package messagequeue

import (
	"errors"
	"strconv"

	"github.com/aws/aws-sdk-go/service/sqs"
)

/*
const (
	deleterDefaultSize = 10
	deleterDefaultTTL  = 2 * time.Second
)

var (
	deleterDefaultParams BufferParameters
)

func init() {
	deleterDefaultParams = BufferParameters{deleterDefaultSize, deleterDefaultTTL}
}
*/

func validateDeleterMessages(messages []Message) error {
	var errs CoalescedError
	for _, message := range messages {
		if message == nil {
			errs = append(errs, errors.New("message is nil"))
			continue
		}
		m, ok := message.(*SQSMessage)
		if !ok {
			errs = append(errs, errors.New("message is not an SQS message"))
			continue
		}
		if m.ReceiptHandle == nil || *m.ReceiptHandle == "" {
			errs = append(errs, errors.New("message receipt handle is empty"))
			continue
		}
	}
	return NewCoalescedError(errs)
}

// Deleter is the common interface for deleting messages that all queue
// implementations support.
type Deleter interface {
	Delete(...Message) error
	Close() error
}

// SQSDeleter deletes messages from an SQS queue.
type SQSDeleter struct {
	queue *SQSQueue
}

// Delete deletes messages from the queue.
func (deleter *SQSDeleter) Delete(messages ...Message) error {
	if err := deleter.canDelete(); err != nil {
		return err
	}
	if err := validateDeleterMessages(messages); err != nil {
		return err
	}
	req := &sqs.DeleteMessageBatchInput{QueueUrl: &deleter.queue.url}
	for i, message := range messages {
		sqsMessage := message.(*SQSMessage)
		id := strconv.Itoa(i)
		req.Entries = append(req.Entries, &sqs.DeleteMessageBatchRequestEntry{
			Id:            &id,
			ReceiptHandle: sqsMessage.ReceiptHandle,
		})
	}
	resp, err := deleter.queue.DeleteMessageBatch(req)
	if err != nil {
		return err
	}
	if len(resp.Failed) > 0 {
		var errs []error
		for _, fail := range resp.Failed {
			errs = append(errs, &BatchResultError{fail})
		}
		return NewCoalescedError(errs)
	}
	return nil
}

func (deleter *SQSDeleter) canDelete() error {
	if deleter.queue == nil {
		return errors.New("queue is closed")
	}
	return nil
}

// Close closes the deleter.
func (deleter *SQSDeleter) Close() error {
	deleter.queue = nil
	return nil
}

/*
type AsyncDeleter interface {
	Deleter
}

type SQSAsyncDeleter struct {
	operationFn bufferedReceivingOperation
}

func (deleter *SQSAsyncDeleter) Send(messages ...*Message) error {
	if deleter.messageChan == nil {
		return errors.New("deleter is closed")
	}
	var errs []error
	for message := range messages {
		if err := validateDeleterMessage(message); err != nil {
			errs = append(errs, err)
		}
	}
	err := CoalesceErrors(errs)
	if err != nil {
		return err
	}
	go func() {
		for message := range messages {
			deleter.messageChan <- m

		}
	}()
	return nil
}

func (deleter *SQSAsyncDeleter) Close() error {
	close(deleter.messageChan)
	var errs []error
	for err := range deleter.errChan {
		errs = append(errs, err)
	}
	return CoalesceErrors(errs)
}
*/
