package messagequeue

import (
	"fmt"
	"strings"

	"github.com/aws/aws-sdk-go/service/sqs"
)

// CoalescedError is a collection of errors coalesced into a single error.
type CoalescedError []error

// NewCoalescedError creates a CoalescedError from a slice of errors.
func NewCoalescedError(errs []error) error {
	if errs == nil { // CoalescedError(nil) != nil
		return nil
	}
	return CoalescedError(errs)
}

// BatchResultError wraps the sqs.BatchResultErrorEntry to implement the error
// interface.
type BatchResultError struct {
	*sqs.BatchResultErrorEntry
}

// Error creates a string representation of the collected errors.
func (errs CoalescedError) Error() string {
	if len(errs) == 0 {
		panic(fmt.Sprintf("invalid CoalescedError %#v", errs))
	}
	var errStrings []string
	for i, err := range errs {
		if err == nil {
			panic(fmt.Sprintf("nil error in CoalescedError[%d] %#v", i, errs))
		}
		errStrings = append(errStrings, err.Error())
	}
	return strings.Join(errStrings, "\n")
}

// Error creates a string represntation of the failed batch result.
func (err *BatchResultError) Error() string {
	if err.BatchResultErrorEntry == nil {
		return ""
	}
	s := []string{"batch request failed"}
	if err.Id != nil {
		s = append(s, "id: "+*err.Id)
	}
	if err.Code != nil {
		s = append(s, "code: "+*err.Code)
	}
	if err.SenderFault != nil {
		s = append(s, fmt.Sprintf("senderFault: %t", *err.SenderFault))
	}
	if err.Message != nil {
		s = append(s, "- "+*err.Message)
	}
	return strings.Join(s, " ")
}
