package messagequeue

import (
	"errors"
	"testing"

	"github.com/aws/aws-sdk-go/service/sqs"
)

type coalescedErrorErrorEntry struct {
	in  []error
	out string
}

func TestCoalescedErrorErrorNil(t *testing.T) {
	c := CoalescedError(nil)
	var str string
	defer func() {
		if r := recover(); r == nil {
			t.Errorf("(%v) Error() = %q wanted panic", c, str)
		}
	}()
	str = c.Error()
}

func TestCoalescedErrorErrorEmpty(t *testing.T) {
	c := CoalescedError{}
	defer func() {
		if r := recover(); r != nil {
		}
	}()
	str := c.Error()
	t.Errorf("(%v) Error() = %q wanted panic", c, str)
}

func TestCoalescedErrorErrorNilElement(t *testing.T) {
	c := CoalescedError{nil}
	defer func() {
		if r := recover(); r != nil {
		}
	}()
	str := c.Error()
	t.Errorf("(%v) Error() = %q wanted panic", c, str)
}

func TestCoalescedErrorError(t *testing.T) {
	var testInput = []coalescedErrorErrorEntry{
		{[]error{errors.New("test")}, "test"},
		{[]error{errors.New("test1"), errors.New("test2")}, "test1\ntest2"},
	}
	for _, e := range testInput {
		testCoalescedErrorErrorHelper(t, e)
	}
}

func testCoalescedErrorErrorHelper(t *testing.T, e coalescedErrorErrorEntry) {
	c := CoalescedError(e.in)
	str := c.Error()
	if str != e.out {
		t.Errorf("(%v) Error() = %q wanted %q", c, str, e.out)
	}
}

type batchResultErrorErrorEntry struct {
	in  *sqs.BatchResultErrorEntry
	out string
}

func TestBatchResultErrorError(t *testing.T) {
	emptyStr := ""
	trueBool := true
	falseBool := false
	idStr := "1"
	codeStr := "100"
	messageStr := "test"
	var testInput = []batchResultErrorErrorEntry{
		{nil, ""},
		{&sqs.BatchResultErrorEntry{}, "batch request failed"},
		{&sqs.BatchResultErrorEntry{}, "batch request failed"},
		{&sqs.BatchResultErrorEntry{Id: &emptyStr}, "batch request failed id: "},
		{&sqs.BatchResultErrorEntry{Id: &idStr}, "batch request failed id: 1"},
		{&sqs.BatchResultErrorEntry{Code: &emptyStr}, "batch request failed code: "},
		{&sqs.BatchResultErrorEntry{Code: &codeStr}, "batch request failed code: 100"},
		{&sqs.BatchResultErrorEntry{Message: &emptyStr}, "batch request failed - "},
		{&sqs.BatchResultErrorEntry{Message: &messageStr}, "batch request failed - test"},
		{&sqs.BatchResultErrorEntry{SenderFault: &trueBool}, "batch request failed senderFault: true"},
		{&sqs.BatchResultErrorEntry{SenderFault: &falseBool}, "batch request failed senderFault: false"},
		{&sqs.BatchResultErrorEntry{Id: &idStr, Code: &codeStr, SenderFault: &falseBool, Message: &messageStr},
			"batch request failed id: 1 code: 100 senderFault: false - test"},
	}
	for _, e := range testInput {
		testBatchResultErrorErrorHelper(t, e)
	}
}

func testBatchResultErrorErrorHelper(t *testing.T, e batchResultErrorErrorEntry) {
	b := &BatchResultError{e.in}
	str := b.Error()
	if str != e.out {
		t.Errorf("(%v) Error() = %q wanted %q", b, str, e.out)
	}
}
