// Package messagequeue provides an interface for generic queue clients and an adapter to AWS SQS.
package messagequeue

import (
	"errors"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/sqs"
	"github.com/aws/aws-sdk-go/service/sqs/sqsiface"
)

// Queue is the interface that wraps factory methods that queue implementations
// support.
type Queue interface {
	Message([]byte) (Message, error)
	Receiver(params BufferParameters) Receiver
	Sender() Sender
	Deleter() Deleter
}

// An SQSQueue is a factory that can create Messages, Receivers, Senders, and
// Deleters for an SQS queue.
type SQSQueue struct {
	sqsiface.SQSAPI
	url string
}

// NewSQSQueue returns a new SQSQueue with name, using credentials from
// awsConfig.
func NewSQSQueue(awsConfig *aws.Config, name string) (*SQSQueue, error) {
	if awsConfig == nil {
		awsConfig = aws.NewConfig()
	}
	if name == "" {
		return nil, errors.New("name")
	}
	q := &SQSQueue{SQSAPI: sqs.New(awsConfig)}

	resp, err := q.GetQueueUrl(&sqs.GetQueueUrlInput{QueueName: &name})
	if err != nil {
		return nil, err
	}
	if resp.QueueUrl == nil || *resp.QueueUrl == "" {
		return nil, errors.New("queue url is empty")
	}
	q.url = *resp.QueueUrl

	return q, nil
}

// TODO: Implement List() method

// Message returns a new Message for queue.
func (q *SQSQueue) Message(b []byte) (*SQSMessage, error) {
	s := string(b)
	m := &SQSMessage{Message: &sqs.Message{Body: &s}}
	return m, nil
}

// Receiver returns a new Receiver for messages from queue.
func (q *SQSQueue) Receiver(params BufferParameters) Receiver {
	return &SQSReceiver{queue: q, bufferConfiguration: &bufferConfiguration{params, receiverDefaultParams}}
}

// Sender returns a new Sender for messages to queue.
func (q *SQSQueue) Sender() Sender {
	sender := &SQSSender{q}
	return sender
}

// Deleter returns a new Deleter for messages from queue.
func (q *SQSQueue) Deleter() Deleter {
	deleter := &SQSDeleter{q}
	return deleter
}
