package messagequeue

import (
	"errors"
	"testing"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/sqs"
)

// TestNewNilAWSConfig requires AWS_REGION environment variable be set in the test runner, and the SQS queue with
// name created.
func TestNewNilAWSConfig(t *testing.T) {
	var awsConfig *aws.Config
	name := "test"
	s, err := NewSQSQueue(awsConfig, name)
	if err != nil {
		t.Errorf("NewSQSQueue(%v, %q) = _, %v, wanted %v", awsConfig, name, err, nil)
	}
	if s == nil {
		t.Fatalf("NewSQSQueue(%v, %q) = %v, _, wanted %T", awsConfig, name, s, &SQSQueue{})
	}
	if s.SQSAPI == nil {
		t.Errorf("NewSQSQueue(%v, %q) = q, _, q.SQSAPI = %v wanted %T", awsConfig, name, s.SQSAPI, &sqs.SQS{})
	}
}

func TestNewSQSQueueEmptyName(t *testing.T) {
	awsConfig := aws.NewConfig()
	name := ""
	sqs, err := NewSQSQueue(nil, name)
	if err == nil {
		t.Errorf("NewSQSQueue(%v, %q) = _, %v, wanted %T", awsConfig, name, err, errors.New(""))
	}
	if sqs != nil {
		t.Errorf("NewSQSQueue(%v, %q) = %v, _, wanted %v", awsConfig, name, sqs, nil)
	}
}

type testQueueMessageEntry struct {
	in  []byte
	out string
}

func TestQueueMessage(t *testing.T) {
	var testInput = []testQueueMessageEntry{
		{nil, ""},
		{[]byte{}, ""},
		{[]byte(""), ""},
		{[]byte("test"), "test"},
	}
	for _, e := range testInput {
		testQueueMessageHelper(t, e)
	}
}

func testQueueMessageHelper(t *testing.T, e testQueueMessageEntry) {
	queue := &SQSQueue{}
	m, err := queue.Message(e.in)
	if m.Message == nil {
		t.Errorf("(%v) Message(%s) = m, _; m.Message = %v, wanted %T", queue, e.in, m.Message, SQSMessage{&sqs.Message{}}.Message)
	}
	if m.Message.Body == nil {
		t.Errorf("(%v) Message(%s) = m, _; m.Message.Body = %v, wanted %T", queue, e.in, m.Message.Body, SQSMessage{&sqs.Message{}}.Message.Body)
	}
	if *m.Message.Body != e.out {
		t.Errorf("(%v) Message(%s) = m, _; *m.Message.Body = %q, wanted %q", queue, e.in, *m.Message.Body, e.out)
	}
	if err != nil {
		t.Errorf("(%v) Message(%s) = _, %v, wanted %v", queue, e.in, err, nil)
	}
}

func TestMessageBytesNilMessage(t *testing.T) {
	message := &SQSMessage{}
	b, err := message.Bytes()
	if b != nil {
		t.Errorf("(%v) Body() = %v, _, wanted %v", message, b, nil)
	}
	if err == nil {
		t.Errorf("(%v) Body() = _, %v, wanted %T", message, err, errors.New(""))
	}
}

func TestMessageBytesNilMessageBody(t *testing.T) {
	message := &SQSMessage{&sqs.Message{}}
	b, err := message.Bytes()
	if b != nil {
		t.Errorf("(%v) Body() = %v, _, wanted %v", message, b, nil)
	}
	if err == nil {
		t.Errorf("(%v) Body() = _, %v, wanted %T", message, err, errors.New(""))
	}
}

func TestMessageBytesString(t *testing.T) {
	str := "test"
	message := &SQSMessage{&sqs.Message{Body: &str}}
	b, err := message.Bytes()
	if string(b) != str {
		t.Errorf("(%v) Body() = %v, _, wanted %v", message, b, []byte(str))
	}
	if err != nil {
		t.Errorf("(%v) Body() = _, %v, wanted %v", message, err, nil)
	}
}

func TestReceiverCanReceiveNewQueue(t *testing.T) {
	r := &SQSReceiver{queue: &SQSQueue{}}
	err := r.canReceive()
	if err != nil {
		t.Errorf("(%v) canReceive() = %v, wanted %v", r, err, nil)
	}
}

func TestReceiverCanReceiveNilQueue(t *testing.T) {
	r := &SQSReceiver{}
	err := r.canReceive()
	if err == nil {
		t.Errorf("(%v) canReceive() = %v, wanted %T", r, err, errors.New(""))
	}
}

func TestReceiverCloseCanReceive(t *testing.T) {
	r := &SQSReceiver{queue: &SQSQueue{}}
	if err := r.Close(); err != nil {
		t.Fatalf("(%v) Close() = %v, wanted %v", r, err, nil)
	}
	err := r.canReceive()
	if err == nil {
		t.Errorf("(%v) canReceive() = %v, wanted %T", r, err, errors.New(""))
	}
}

func TestReceiverGetParams(t *testing.T) {
	expectedSize := 10
	expectedTTL := 100 * time.Second

	r := &SQSReceiver{bufferConfiguration: &bufferConfiguration{BufferParameters{expectedSize, expectedTTL}, receiverDefaultParams}}
	size, ttl := r.getParams()
	if size != expectedSize {
		t.Errorf("(%v) getParams() = %v, _, wanted %v", r, size, expectedSize)
	}
	if ttl != expectedTTL {
		t.Errorf("(%v) getParams() = _, %v, wanted %v", r, ttl, expectedTTL)
	}
}

func TestReceiverGetParamsNilParams(t *testing.T) {
	expectedSize := receiverDefaultSize
	expectedTTL := receiverDefaultTTL

	r := &SQSReceiver{}
	size, ttl := r.getParams()
	if size != expectedSize {
		t.Errorf("(%v) getParams() = %v, _, wanted %v", r, size, expectedSize)
	}
	if ttl != expectedTTL {
		t.Errorf("(%v) getParams() = _, %v, wanted %v", r, ttl, expectedTTL)
	}
}

/*
func TestValidateSenderMessagesNilMessages(t *testing.T) {

}

func TestValidateSenderMessagesEmptyMessages(t *testing.T) {

}
*/

type TestMessage []byte

func (t TestMessage) Bytes() ([]byte, error) {
	return t, nil
}

func TestValidateSenderMessagesNilMessage(t *testing.T) {
	msgs := []Message{nil}
	err := validateSenderMessages(msgs)
	if err == nil {
		t.Errorf("validateSenderMessages(%v) = %v, wanted %T", msgs, err, errors.New(""))
	}
}

func TestValidateSenderMessagesNonSQSMessage(t *testing.T) {
	msgs := []Message{TestMessage{}}
	err := validateSenderMessages(msgs)
	if err == nil {
		t.Errorf("validateSenderMessages(%v) = %v, wanted %T", msgs, err, errors.New(""))
	}
}

func TestValidateSenderMessagesNilBody(t *testing.T) {
	msgs := []Message{&SQSMessage{Message: &sqs.Message{}}}
	err := validateSenderMessages(msgs)
	if err == nil {
		t.Errorf("validateSenderMessages(%v) = %v, wanted %T", msgs, err, errors.New(""))
	}
}

func TestValidateSenderMessagesEmptyBody(t *testing.T) {
	body := ""
	msgs := []Message{&SQSMessage{Message: &sqs.Message{Body: &body}}}
	err := validateSenderMessages(msgs)
	if err == nil {
		t.Errorf("validateSenderMessages(%v) = %v, wanted %T", msgs, err, errors.New(""))
	}
}

func TestValidateSenderMessagesSingleMessage(t *testing.T) {
	body := "test"
	msgs := []Message{&SQSMessage{Message: &sqs.Message{Body: &body}}}
	err := validateSenderMessages(msgs)
	if err != nil {
		t.Errorf("validateSenderMessages(%v) = %v, wanted %T", msgs, err, nil)
	}
}

func TestValidateSenderMessagesMultipleMessage(t *testing.T) {
	body := "test"
	msgs := []Message{&SQSMessage{Message: &sqs.Message{Body: &body}},
		&SQSMessage{Message: &sqs.Message{Body: &body}}}
	err := validateSenderMessages(msgs)
	if err != nil {
		t.Errorf("validateSenderMessages(%v) = %v, wanted %T", msgs, err, nil)
	}
}

func TestValidateSenderMessagesMultipleMessageErrors(t *testing.T) {
	msgs := []Message{nil, nil}
	err := validateSenderMessages(msgs)
	if err == nil {
		t.Fatalf("validateSenderMessages(%v) = %v, wanted %T", msgs, err, CoalescedError{})
	}
	if _, ok := err.(CoalescedError); !ok {
		t.Errorf("validateSenderMessages(%v) = %v, wanted %T", msgs, err, CoalescedError{})
	}
}

func TestSenderCanSendNewQueue(t *testing.T) {
	s := &SQSSender{queue: &SQSQueue{}}
	err := s.canSend()
	if err != nil {
		t.Errorf("(%v) canSend() = %v, wanted %v", s, err, nil)
	}
}

func TestSenderCanSendNilQueue(t *testing.T) {
	s := &SQSSender{}
	err := s.canSend()
	if err == nil {
		t.Errorf("(%v) canSend() = %v, wanted %T", s, err, errors.New(""))
	}
}

func TestSenderCloseCanSend(t *testing.T) {
	s := &SQSSender{queue: &SQSQueue{}}
	if err := s.Close(); err != nil {
		t.Fatalf("(%v) Close() = %v, wanted %v", s, err, nil)
	}
	err := s.canSend()
	if err == nil {
		t.Errorf("(%v) canReceive() = %v, wanted %T", s, err, errors.New(""))
	}
}

func TestValidateDeleterMessagesNilMessage(t *testing.T) {
	msgs := []Message{nil}
	err := validateDeleterMessages(msgs)
	if err == nil {
		t.Errorf("validateDeleterMessages(%v) = %v, wanted %T", msgs, err, errors.New(""))
	}
}

func TestValidateDeleterMessagesNonSQSMessage(t *testing.T) {
	msgs := []Message{TestMessage{}}
	err := validateDeleterMessages(msgs)
	if err == nil {
		t.Errorf("validateDeleterMessages(%v) = %v, wanted %T", msgs, err, errors.New(""))
	}
}

func TestValidateDeleterMessagesNilBody(t *testing.T) {
	msgs := []Message{&SQSMessage{Message: &sqs.Message{}}}
	err := validateDeleterMessages(msgs)
	if err == nil {
		t.Errorf("validateDeleterMessages(%v) = %v, wanted %T", msgs, err, errors.New(""))
	}
}

func TestValidateDeleterMessagesEmptyBody(t *testing.T) {
	receiptHandle := ""
	msgs := []Message{&SQSMessage{Message: &sqs.Message{ReceiptHandle: &receiptHandle}}}
	err := validateDeleterMessages(msgs)
	if err == nil {
		t.Errorf("validateDeleterMessages(%v) = %v, wanted %T", msgs, err, errors.New(""))
	}
}

func TestValidateDeleterMessagesSingleMessage(t *testing.T) {
	receiptHandle := "test"
	msgs := []Message{&SQSMessage{Message: &sqs.Message{ReceiptHandle: &receiptHandle}}}
	err := validateDeleterMessages(msgs)
	if err != nil {
		t.Errorf("validateDeleterMessages(%v) = %v, wanted %T", msgs, err, nil)
	}
}

func TestValidateDeleterMessagesMultipleMessage(t *testing.T) {
	receiptHandle := "test"
	msgs := []Message{&SQSMessage{Message: &sqs.Message{ReceiptHandle: &receiptHandle}},
		&SQSMessage{Message: &sqs.Message{ReceiptHandle: &receiptHandle}}}
	err := validateDeleterMessages(msgs)
	if err != nil {
		t.Errorf("validateDeleterMessages(%v) = %v, wanted %T", msgs, err, nil)
	}
}

func TestValidateDeleterMessagesMultipleMessageErrors(t *testing.T) {
	msgs := []Message{nil, nil}
	err := validateDeleterMessages(msgs)
	if err == nil {
		t.Fatalf("validateDeleterMessages(%v) = %v, wanted %T", msgs, err, CoalescedError{})
	}
	if _, ok := err.(CoalescedError); !ok {
		t.Errorf("validateDeleterMessages(%v) = %v, wanted %T", msgs, err, CoalescedError{})
	}
}

func TestDeleterCanDeleteNewQueue(t *testing.T) {
	d := &SQSDeleter{queue: &SQSQueue{}}
	err := d.canDelete()
	if err != nil {
		t.Errorf("(%v) canSend() = %v, wanted %v", d, err, nil)
	}
}

func TestDeleterCanDeleteNilQueue(t *testing.T) {
	d := &SQSDeleter{}
	err := d.canDelete()
	if err == nil {
		t.Errorf("(%v) canSend() = %v, wanted %T", d, err, errors.New(""))
	}
}

func TestDeleterCloseCanDelete(t *testing.T) {
	d := &SQSDeleter{queue: &SQSQueue{}}
	if err := d.Close(); err != nil {
		t.Fatalf("(%v) Close() = %v, wanted %v", d, err, nil)
	}
	err := d.canDelete()
	if err == nil {
		t.Errorf("(%v) canReceive() = %v, wanted %T", d, err, errors.New(""))
	}
}
