package messagequeue

import (
	"errors"
	"time"

	"github.com/aws/aws-sdk-go/service/sqs"
)

const (
	receiverDefaultSize = 10
	receiverDefaultTTL  = 2 * time.Second
)

var (
	receiverDefaultParams BufferParameters
)

func init() {
	receiverDefaultParams = BufferParameters{receiverDefaultSize, receiverDefaultTTL}
}

// Receiver is the common interface for receiving messages that all queue
// implementations support.
type Receiver interface {
	Receive() ([]Message, error)
	Close() error
}

// SQSReceiver receives messages from an SQS queue.
type SQSReceiver struct {
	queue               *SQSQueue
	bufferConfiguration *bufferConfiguration
}

// Receive receives messages from the queue.
func (receiver *SQSReceiver) Receive() ([]Message, error) {
	if err := receiver.canReceive(); err != nil {
		return nil, err
	}
	size, ttl := receiver.getParams()
	s := int64(size)
	t := int64(ttl.Seconds())
	resp, err := receiver.queue.ReceiveMessage(&sqs.ReceiveMessageInput{QueueUrl: &receiver.queue.url, MaxNumberOfMessages: &s, WaitTimeSeconds: &t})
	if err != nil {
		return nil, err
	}
	messages := make([]Message, 0, len(resp.Messages))
	for _, message := range resp.Messages {
		messages = append(messages, &SQSMessage{message})
	}
	return messages, nil
}

func (receiver *SQSReceiver) canReceive() error {
	if receiver.queue == nil {
		return errors.New("queue is closed")
	}
	return nil
}

func (receiver *SQSReceiver) getParams() (size int, ttl time.Duration) {
	if receiver.bufferConfiguration == nil {
		receiver.bufferConfiguration = &bufferConfiguration{defaults: receiverDefaultParams}
	}
	return receiver.bufferConfiguration.getParams()
}

// Close closes the receiver.
func (receiver *SQSReceiver) Close() error {
	// receiver always closes successfully
	receiver.queue = nil
	return nil
}
