package messagequeue

import (
	"errors"
	"strconv"

	"github.com/aws/aws-sdk-go/service/sqs"
)

/*
const (
	senderDefaultSize = 10
	senderDefaultTTL  = 2 * time.Second
)

var (
	senderDefaultParams BufferParameters
)

func init() {
	senderDefaultParams = BufferParameters{senderDefaultSize, senderDefaultTTL}
}
*/

func validateSenderMessages(messages []Message) error {
	var errs CoalescedError
	for _, message := range messages {
		if message == nil {
			errs = append(errs, errors.New("message is nil"))
			continue
		}
		m, ok := message.(*SQSMessage)
		if !ok {
			errs = append(errs, errors.New("message is not an SQS message"))
			continue
		}
		if m.Body == nil || *m.Body == "" {
			errs = append(errs, errors.New("message body is empty"))
			continue
		}
	}
	return NewCoalescedError(errs)
}

// Sender is the common interface for sending messages that all queue
// implementations support.
type Sender interface {
	Send(...Message) error
	Close() error
}

// SQSSender sends messages to an SQS queue.
type SQSSender struct {
	queue *SQSQueue
}

// Send sends messages to the queue.
func (sender *SQSSender) Send(messages ...Message) error {
	if err := sender.canSend(); err != nil {
		return err
	}
	if err := validateSenderMessages(messages); err != nil {
		return err
	}
	req := &sqs.SendMessageBatchInput{QueueUrl: &sender.queue.url}
	for i, message := range messages {
		sqsMessage := message.(*SQSMessage)
		id := strconv.Itoa(i)
		req.Entries = append(req.Entries, &sqs.SendMessageBatchRequestEntry{
			MessageBody: sqsMessage.Body,
			Id:          &id,
		})
	}
	resp, err := sender.queue.SendMessageBatch(req)
	if err != nil {
		return err
	}
	if len(resp.Failed) > 0 {
		var errs []error
		for _, fail := range resp.Failed {
			errs = append(errs, &BatchResultError{fail})
		}
		return NewCoalescedError(errs)
	}
	return nil
}

func (sender *SQSSender) canSend() error {
	if sender.queue == nil {
		return errors.New("queue is closed")
	}
	return nil
}

// Close closes the sender.
func (sender *SQSSender) Close() error {
	sender.queue = nil
	return nil
}

/*
type SQSAsyncSender struct {
	operation bufferedReceivingOperation
}

func (sender *SQSAsyncSender) Send(messages ...*Message) error {
	if sender.messageChan == nil {
		return errors.New("sender is closed")
	}
	var errs []error
	for message := range messages {
		if err := validateSenderMessage(message); err != nil {
			errs = append(errs, err)
		}
	}
	err := CoalesceErrors(errs)
	if err != nil {
		return err
	}
	go func() {
		for m := range messages {
			sender.operation.messageChan <- m
		}
	}()
	return nil
}

func (sender *SQSAsyncSender) Close() error {
	close(sender.messageChan)
	var errs []error
	for err := range sender.operation.errChan {
		errs = append(errs, err)
	}
	return CoalesceErrors(errs)
}
*/
