package interchange

import (
	"encoding/json"
	"fmt"
	"io/ioutil"
	"log"
	"path"
	"strings"

	api "github.com/hashicorp/consul/api"
)

const (
	// EnvName is the OS Environment Variable specifying where the forerunner
	// data file will loaded from.
	EnvName = "FORERUNNER_INTERCHANGE"
	// Version is the current version of the forerunner data file.
	Version = "0.3.0"
)

type EntrySet map[string]*ConfigEntry
type Source string

const (
	ENV     Source = "env"
	Flags          = "flags"
	Config         = "config"
	Consul         = "consul"
	Default        = "default"
)

type Interchange struct {
	Version string
	Entries map[string]*ConfigEntry
}

type ConfigEntry struct {
	Value  interface{}
	Source Source
	Secret bool
}

// LoadInterchange will load the config return the location of a processed output.
func LoadInterchange(appName, environment string) (string, error) {
	if appName == "" || environment == "" {
		return "", fmt.Errorf("Invalid arguments to LoadInterchange")
	}

	f, err := ioutil.TempFile("", appName)
	if err != nil {
		return "", nil
	}

	// TODO:
	//
	// * load config file vars
	config := &Interchange{
		Version: Version,
		Entries: merge(
			nil,
			getConsul(appName, environment),
		),
	}

	err = json.NewEncoder(f).Encode(&config)
	if err != nil {
		return "", err
	}

	if err := f.Sync(); err != nil {
		return "", err
	}

	return f.Name(), nil
}

func merge(configVars, consulVars map[string]*ConfigEntry) map[string]*ConfigEntry {
	res := map[string]*ConfigEntry{}
	for _, entries := range []map[string]*ConfigEntry{
		// These are in order from least to most important to merge correctly!
		consulVars,
		configVars,
	} {
		for k, v := range entries {
			res[k] = v
		}
	}

	return res
}

func getConsul(appName, environment string) map[string]*ConfigEntry {
	res := map[string]*ConfigEntry{}
	prefix := path.Join("settings", appName, environment) + "/"

	// TODO: don't hard code the consul api endpoint
	client, _ := api.NewClient(
		&api.Config{
			Address: "consul.internal.justin.tv",
		})

	kvs, _, err := client.KV().List(prefix, nil)
	// TODO: depending upon the error this should bubble up an error and load
	// the cached copy.
	if err != nil {
		log.Printf("Error fetching from consul: %v", err)
		return res
	}

	if kvs == nil {
		log.Printf("Got no settings from consul: %q", prefix)
	}

	for _, kv := range kvs {
		if kv.Key == prefix {
			continue
		}

		key := strings.TrimPrefix(kv.Key, prefix)
		res[key] = &ConfigEntry{
			Value:  string(kv.Value),
			Source: Consul,
		}
	}

	return res
}
