package twitchconsulapi

import (
	"fmt"
	"net/http"
	"net/http/httptest"
	"net/url"
	"testing"
)

func fakeConsul() *httptest.Server {
	mux := http.NewServeMux()
	mux.HandleFunc("/v1/catalog/datacenters", fakeDatacenters)
	mux.HandleFunc("/v1/catalog/service/nodeinfo", fakeNodeInfo)
	mux.HandleFunc("/v1/catalog/service/nodeinfo2", fakeNodeInfo2)
	mux.HandleFunc("/v1/health/state/passing", fakeStatusInfo)

	ts := httptest.NewServer(mux)
	return ts
}

func fakeDatacenters(w http.ResponseWriter, r *http.Request) {
	fmt.Fprintf(w, `["test1", "test2", "test3"]`)
}

func fakeStatusInfo(w http.ResponseWriter, r *http.Request) {
	dc := r.URL.Query().Get("dc")

	switch dc {
	case "test1":
		fmt.Fprintf(w, `[
		{"ServiceName": "Serf Health", "ServiceID": "serfHealth", "Output": "", "Notes": "",
		 "Status": "passing", "Name": "Serf Health", "CheckID": "serfHealth", "Node": "staging-17-app-0.dev"}
		]`)
	case "test2":
		fmt.Fprintf(w, `[]`)
	case "test3":
		w.WriteHeader(http.StatusInternalServerError)
	}
}

func genServiceInfo(host, ip, dc, service string) string {
	return fmt.Sprintf(`[
		{"ServicePort": 0, "ServiceAddress": "", "ServiceTags": ["fqdn=%s.%s", "lsbdistcodename=precise"],
    "ServiceName": "%s", "ServiceID": "%s", "Address": "%s", "Node": "%s"}
		]`, host, dc, service, service, ip, host)
}

func fakeNodeInfo(w http.ResponseWriter, r *http.Request) {
	dc := r.URL.Query().Get("dc")

	switch dc {
	case "test1":
		fmt.Fprintf(w, genServiceInfo("staging-17-app-0.dev", "127.1.6.32", "test1.justin.tv", "nodeinfo"))
	case "test2":
		fmt.Fprintf(w, genServiceInfo("staging-16-app-0.dev", "127.1.5.18", "test2.justin.tv", "nodeinfo"))
	case "test3":
		w.WriteHeader(http.StatusInternalServerError)
	}
}

func fakeNodeInfo2(w http.ResponseWriter, r *http.Request) {
	dc := r.URL.Query().Get("dc")

	switch dc {
	case "test1":
		fmt.Fprintf(w, genServiceInfo("staging-17-app-0.dev", "127.1.6.32", "test1.justin.tv", "nodeinfo2"))
	case "test2":
		fmt.Fprintf(w, genServiceInfo("staging-16-app-0.dev", "127.1.5.18", "test2.justin.tv", "nodeinfo2"))
	case "test3":
		w.WriteHeader(http.StatusInternalServerError)
	}
}

func fakeServiceInfo(w http.ResponseWriter, r *http.Request) {
	dc := r.URL.Query().Get("dc")

	switch dc {
	case "test1":
		fmt.Fprintf(w, `[{
    "ServicePort": 0, "ServiceAddress": "",
    "ServiceTags": ["fqdn=staging-17-app-0.dev.test1.justin.tv", "lsbdistcodename=precise"],
    "ServiceName": "nodeinfo", "ServiceID": "nodeinfo", "Address": "10.192.130.47", "Node": "staging-17-app-0.dev"}]`)
	case "test2":
		fmt.Fprintf(w, `[]`)
	case "test3":
		w.WriteHeader(http.StatusInternalServerError)
	}
}

func TestParseFQDN(t *testing.T) {
	tags := []string{"fqdn=reltool1.sfo01.justin.tv", "lsbdistcodename=precise"}

	fqdn, err := parseFQDN(tags)
	if err != nil {
		t.Fatal(err)
	}

	if fqdn != "reltool1.sfo01.justin.tv" {
		t.Fatalf("expected: reltool1.sfo01.justin.tv, got: %v", fqdn)
	}
}

// TestGetAliveHostnames is a first swipe at adding tests for GetAliveHostnames.
// Due to the difficulty of faking consul this is actually testing a bunch of things:
//
// 1. A datacenter that doesn't have a leader (returns 500).
// 2. A datacenter that has a server that is alive.
// 3. A datacenter that has a server that is not alive.
func TestGetAliveHostnames(t *testing.T) {
	ts := fakeConsul()
	u, err := url.Parse(ts.URL)
	if err != nil {
		t.Fatal(err)
	}

	client, err := NewClient(u.Host, nil)
	if err != nil {
		t.Fatal(err)
	}

	hosts, err := GetAliveHostnames(client, []string{"nodeinfo", "nodeinfo2"}, "")
	if err != nil {
		t.Fatal(err)
	}

	expected := "staging-17-app-0.dev.test1.justin.tv"
	t.Logf("Hosts: %#v", hosts)
	if len(hosts) != 1 || (len(hosts) > 0 && hosts[0] != expected) {
		t.Errorf(`GetAliveHostnames("nodeinfo", "nodeinfo2") = %#v; want: %#v`, hosts, []string{expected})
	}

	defer ts.Close()
}
