package twitchconsulapi

import (
	"log"

	"github.com/hashicorp/consul/api"
)

type Datacenter struct {
	FQDNs      map[string]string
	Deployable map[string]bool
	Alive      map[string]bool
	Name       string
	client     *api.Client
	queryParms *api.QueryOptions
}

func NewDatacenter(name string, client *api.Client) (*Datacenter, error) {
	return &Datacenter{
		Name:       name,
		FQDNs:      map[string]string{},
		Deployable: map[string]bool{},
		Alive:      map[string]bool{},
		client:     client,
		queryParms: &api.QueryOptions{Datacenter: name},
	}, nil
}

// GetData will fetch basic data from the datacenter. For example it will get a
// list of all FQDNs for the datacenter for later processing.
func (dc *Datacenter) GetData() error {
	if err := dc.getFQDNs(); err != nil {
		return err
	}

	if err := dc.getAlive(); err != nil {
		return err
	}

	return nil
}

func (dc *Datacenter) getFQDNs() error {
	services, _, err := dc.client.Catalog().Service("nodeinfo", "", dc.queryParms)
	if err != nil {
		return err
	}

	for _, service := range services {
		if fqdn, err := parseFQDN(service.ServiceTags); err == nil {
			dc.FQDNs[service.Node] = fqdn
		} else {
			log.Println(err)
		}
	}

	return nil
}

func (dc *Datacenter) getAlive() error {
	passingChecks, _, err := dc.client.Health().State("passing", dc.queryParms)
	if err != nil {
		return err
	}

	for _, check := range passingChecks {
		if check.CheckID == "serfHealth" {
			dc.Alive[check.Node] = true
		}
	}

	return nil
}

func (dc *Datacenter) getDeployable() error {
	return nil
}

// LookupService will return a list of hosts that are running service and tag.
func (dc *Datacenter) LookupService(service, tag string) ([]string, error) {
	services, _, err := dc.client.Catalog().Service(service, tag, dc.queryParms)
	if err != nil {
		return nil, err
	}

	hostnames := []string{}
	for _, service := range services {
		fqdn, ok := dc.FQDNs[service.Node]
		if !ok {
			log.Printf("Unable to find FQDN for %q", service.Node)
			continue
		}
		if _, ok := dc.Alive[service.Node]; !ok {
			log.Printf("Node %q is offline", service.Node)
			continue
		}
		hostnames = append(hostnames, fqdn)
	}

	return hostnames, nil
}
