package main

import (
	"encoding/json"
	"fmt"
	"net"
	"net/http"
	"os"
	"path"
	"strconv"
	"strings"

	log "github.com/Sirupsen/logrus"
	_ "github.com/lib/pq"
	"github.com/stvp/rollbar"
	"golang.org/x/net/context"

	"code.justin.tv/common/chitin"
	"code.justin.tv/common/golibs/statsd"
	"code.justin.tv/release/libforerunner"
	"code.justin.tv/release/maelstrom/pkg/config"
	deploy "code.justin.tv/release/maelstrom/pkg/deploy"
	"code.justin.tv/release/smtp"
	"code.justin.tv/release/twitch-consul-api"
	"code.justin.tv/release/twitchstats"
	"github.com/afex/hystrix-go/hystrix"
	"github.com/gorilla/handlers"
	"github.com/gorilla/mux"
	consulapi "github.com/hashicorp/consul/api"
	"github.com/jmoiron/sqlx"
)

var (
	smtpClient   *smtp.Client
	consulClient *consulapi.Client
	statsdClient statsd.Stats
	GitCommit    = "development"
	fr           *libforerunner.Forerunner
	mixpanel     *twitchstats.Stats
)

type Config struct {
	ConsulPrefix       string `description:"Prefix for consul data storage"`
	PgConnInfo         string `description:"Connection string for postgres"`
	Port               int    `description:"Port to bind to for HTTP API"`
	StatsdHost         string `description:"Host to send statsd data to"`
	StatsdPort         int    `description:"Port to send statsd data to"`
	StatsdPrefix       string `description:"Prefix for statsd deploy marks"`
	RollbarToken       string `forerunner:",secret" description:"Token for talking to rollbar"`
	Environment        string `description:"Environment maelstrom is running in"`
	AmazonSQSURL       string `description:"URL for Amazon SQS Queue"`
	AmazonSQSResource  string `description:"Resource Name for Amazon SQS Queue"`
	SMTPServerHostname string `description:"Hostname of SMTP server"`
	SMTPServerPort     int    `description:"Port of SMTP server"`
	MixpanelToken      string `forerunner:",secret" description:"Token for talking to Mixpanel"`
}

var c *Config

func init() {
	var err error

	c = &Config{
		ConsulPrefix:      "application-data/maelstrom/dev",
		Port:              8080,
		PgConnInfo:        "sslmode=disable",
		StatsdPort:        8135,
		RollbarToken:      "9c22637d84314a28b3d4afe9cf92ea85",
		StatsdPrefix:      "maelstrom.development",
		Environment:       "development",
		AmazonSQSURL:      "https://sqs.us-east-1.amazonaws.com/673385534282/maelstrom-dev",
		AmazonSQSResource: "arn:aws:sqs:us-east-1:673385534282:maelstrom-dev",
	}

	log.SetFormatter(&log.JSONFormatter{})

	fr, err = libforerunner.Init(&libforerunner.Options{
		DefaultConfig: c,
	})
	if err != nil {
		log.Fatal(err)
	}

	err = fr.GetConfig(c)
	if err != nil {
		log.Fatal(err)
	}

	fmt.Fprintf(os.Stderr, "Running commit %q with config:\n", GitCommit)
	fr.Print()

	smtpClient, err = smtp.NewClient(c.SMTPServerHostname, c.SMTPServerPort)
	if err != nil {
		log.Fatalf("Can't start SMTP client: %v", err)
	}

	// twitchconsulapi used to be hardcoded to "consul.internal.justin.tv", we
	// should switch it but switching is non-trivial as the queue data is all
	// in SFO.
	consulClient, err = twitchconsulapi.NewClient("consul.internal.justin.tv", chitin.Client(context.Background()))
	if err != nil {
		log.Fatalf("error creating consulapi client: %v", err)
	}

	// Consul is liberal in what it accepts as a valid key but specific in what
	// it returns. By cleaning and removing any leading '/' we can make our
	// paths match what it returns.
	c.ConsulPrefix = strings.TrimLeft(path.Clean(c.ConsulPrefix), "/")

	mixpanel = &twitchstats.Stats{
		MixPanelToken: c.MixpanelToken,
	}
	mixpanel.Client = chitin.Client(context.Background())
	mixpanel.Verbose = true
	mixpanel.Add("system", "maelstrom")
	mixpanel.Add("system_version", GitCommit)
	mixpanel.Add("system_env", c.Environment)

	// setup statsd
	statsdAddr := net.JoinHostPort(c.StatsdHost, strconv.Itoa(c.StatsdPort))
	if c.StatsdPrefix == "" {
		c.StatsdPrefix = strings.Join([]string{"maelstrom", c.Environment}, ".")
	}
	statsdClient, err = statsd.Dial("udp", statsdAddr, statsd.StatsConfig{
		Rate:   1.0,
		Prefix: c.StatsdPrefix,
	})
	if err != nil {
		log.Fatalf("error creating statsd client: %v", err)
	}
	config.StatsdClient = statsdClient

	// Configure Rollbar
	rollbar.Token = c.RollbarToken
	rollbar.Environment = c.Environment
	rollbar.CodeVersion = GitCommit

}

// Configure hystrix settings
func init() {
	hystrix.DefaultMaxConcurrent = 500
	hystrix.DefaultTimeout = 10 * 1000
}

func healthCheck(w http.ResponseWriter, req *http.Request) {
	type hcStruct struct {
		GitCommit string
	}
	if err := json.NewEncoder(w).Encode(&hcStruct{
		GitCommit: GitCommit,
	}); err != nil {
		log.Printf("Error encoding healthCheck: %v", err)
	}
}

func main() {
	router := mux.NewRouter()
	config.CreateHandler(router, "/healthcheck", healthCheck, nil)
	db, err := sqlx.Open("postgres", c.PgConnInfo)
	if err != nil {
		log.Fatal(err)
	}
	defer db.Close()
	deploy.RegisterHandlers(router, db)

	hystrixStreamHandler := hystrix.NewStreamHandler()
	hystrixStreamHandler.Start()
	go http.ListenAndServe(net.JoinHostPort("", "7000"), hystrixStreamHandler)

	if err := db.Ping(); err != nil {
		log.Fatalf("Can't connect to db: %v", err)
	}

	bind := net.JoinHostPort("", strconv.Itoa(c.Port))
	log.Printf("Listening on %q", bind)
	log.Fatal(http.ListenAndServe(
		bind,
		handlers.CombinedLoggingHandler(os.Stdout,
			chitin.Handler(
				router,
				chitin.SetBaseContext(mustTraceContext()),
			),
		),
	))
}

func mustTraceContext() context.Context {
	ctx, err := chitin.ExperimentalTraceContext(context.Background())
	if err != nil {
		log.Fatal(err)
	}

	return ctx
}
