package deploy

import (
	"encoding/json"
	"errors"
	"net/http"

	"code.justin.tv/release/maelstrom/pkg/config"
	jobqueue "code.justin.tv/release/maelstrom/pkg/jobqueue"
	"github.com/gorilla/mux"
	"github.com/jmoiron/sqlx"
)

var (
	db *sqlx.DB
	jq *jobqueue.JobQueue
)

func RegisterHandlers(r *mux.Router, dbInstance *sqlx.DB) {
	db = dbInstance
	jq = jobqueue.NewJobQueue(db)

	//  - List all active jobs
	config.CreateHandler(r, "/v1/deploy/current",
		CurrentListHandler,
		&config.RouteOptions{AddCORS: true}).Methods("GET")

	//  - List all active jobs for a particular repository
	config.CreateHandler(r, "/v1/deploy/current/{repo:[A-Za-z0-9_.-]+}",
		CurrentListHandler,
		&config.RouteOptions{AddCORS: true}).Methods("GET")

	//  - List all active jobs for a particular repository+environment
	config.CreateHandler(r, "/v1/deploy/current/{repo:[A-Za-z0-9_.-]+}/{env:[A-Za-z0-9_.-]+}",
		CurrentListHandler,
		&config.RouteOptions{AddCORS: true}).Methods("GET")

	//  - List all historical jobs
	config.CreateHandler(r, "/v1/deploy/history",
		HistoryListHandler,
		&config.RouteOptions{AddCORS: true}).Methods("GET")

	//  - List all historical jobs for a particular repository
	config.CreateHandler(r, "/v1/deploy/history/{repo:[A-Za-z0-9_.-]+}",
		HistoryListHandler,
		&config.RouteOptions{AddCORS: true}).Methods("GET")

	//  - List all historical jobs for a particular repository+environment
	config.CreateHandler(r, "/v1/deploy/history/{repo:[A-Za-z0-9_.-]+}/{env:[A-Za-z0-9_.-]+}",
		HistoryListHandler,
		&config.RouteOptions{AddCORS: true}).Methods("GET")

	//  - Get current state of a particular deployment job by ID
	config.CreateHandler(r, "/v1/deploy/verify/{id:[a-z0-9]+}",
		GetJobHandler,
		&config.RouteOptions{AddCORS: true}).Methods("GET")

	//  - Provide verification that deployment can proceed to next stage
	config.CreateHandler(r, "/v1/deploy/verify/{id:[a-z0-9]+}",
		VerifyJobHandler,
		&config.RouteOptions{AddCORS: true}).Methods("PUT")

	//  - Rollback the current deployment
	config.CreateHandler(r, "/v1/deploy/rollback/{id:[a-z0-9]+}",
		RollbackJobHandler,
		&config.RouteOptions{AddCORS: true}).Methods("PUT")

	//   - Cancel orchestration job by unique ID
	config.CreateHandler(r, "/v1/deploy/{id:[a-z0-9]+}",
		CancelJobHandler,
		&config.RouteOptions{AddCORS: true}).Methods("DELETE")

	//   - Create a new orchestration job for a repository+environment
	config.CreateHandler(r, "/v1/deploy/{repo:[A-Za-z0-9_.-]+}/{env:[A-Za-z0-9_.-]+}",
		CreateJobHandler,
		&config.RouteOptions{AddCORS: true}).Methods("POST")
}

func CurrentListHandler(w http.ResponseWriter, r *http.Request) {
	// TODO - add hystrix wrappers
	options := &jobqueue.ListJobOptions{}
	vars := mux.Vars(r)
	repo, ok := vars["repo"]
	if ok {
		options.Repository = &repo
	}
	env, oke := vars["env"]
	if oke {
		options.Environment = &env
	}
	jobs, err := jq.List(options)
	if err != nil {
		config.JSONError(w, http.StatusInternalServerError, "error listing jobs", err)
		return
	}
	json.NewEncoder(w).Encode(jobs)
}
func HistoryListHandler(w http.ResponseWriter, r *http.Request) {
	config.JSONError(w, http.StatusNotImplemented, "Not implemented yet", errors.New("err"))
}
func GetJobHandler(w http.ResponseWriter, r *http.Request) {
	vars := mux.Vars(r)
	uuid, _ := vars["uuid"]
	job, err := jq.Peek(uuid)
	if err != nil {
		config.JSONError(w, http.StatusInternalServerError, "error getting job by uuid", err)
		return
	}
	json.NewEncoder(w).Encode(job)
}
func VerifyJobHandler(w http.ResponseWriter, r *http.Request) {
	config.JSONError(w, http.StatusNotImplemented, "Not implemented yet", errors.New("err"))
}
func RollbackJobHandler(w http.ResponseWriter, r *http.Request) {
	config.JSONError(w, http.StatusNotImplemented, "Not implemented yet", errors.New("err"))
}
func CancelJobHandler(w http.ResponseWriter, r *http.Request) {
	config.JSONError(w, http.StatusNotImplemented, "Not implemented yet", errors.New("err"))
}
func CreateJobHandler(w http.ResponseWriter, r *http.Request) {
	config.JSONError(w, http.StatusNotImplemented, "Not implemented yet", errors.New("err"))
}
