package jobs

import (
	"crypto/rand"
	"encoding/hex"
	"encoding/json"
	"errors"
	"fmt"
	"time"

	"code.justin.tv/common/messagequeue"

	log "github.com/Sirupsen/logrus"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/jmoiron/sqlx"
	"github.com/lann/squirrel"
)

type JobQueue struct {
	DB *sqlx.DB
}

type Job struct {
	ID                *int64     `json:"id,omitempty" db:"id"`
	UUID              *string    `json:"id,omitempty" db:"uuid"`
	Repository        *string    `json:"repository,omitempty" db:"repository"`
	SHA               *string    `json:"sha,omitempty" db:"sha"`
	Rollback_Strategy *string    `json:"rollback_strategy,omitempty" db:"rollbackstrategy"`
	Discovery         *string    `json:"rollback_strategy,omitempty" db:"discovery"`
	Deployment_Style  *string    `json:"deployment_style,omitempty" db:"deploymentstyle"`
	Environment       *string    `json:"environment,omitempty" db:"environment"`
	Creator           *string    `json:"creator,omitempty" db:"creator"`
	Status            *string    `json:"state,omitempty" db:"status"`
	CreatedAt         *time.Time `json:"created_at,omitempty" db:"createdat"`
	UpdatedAt         *time.Time `json:"updated_at,omitempty" db:"updatedat"`
}

type ListJobOptions struct {
	Repository  *string
	Environment *string
	UUID        *string
}

type Queue struct {
	Q *messagequeue.SQSQueue
	R *messagequeue.Receiver
	S *messagequeue.Sender
}

type queueLogger struct {
	logger *log.Logger
}

func NewQueueLogger() aws.Logger {
	return &queueLogger{logger: log.New()}
}

func (l queueLogger) Log(args ...interface{}) {
	l.logger.Println(args...)
}

// initialize a new queue connection
func NewQueue(accesskey string, secretkey string, region_name string, resource string) (*Queue, error) {
	c := aws.NewConfig()
	c = c.WithRegion(region_name)
	c = c.WithEndpoint("")
	c = c.WithLogger(NewQueueLogger())
	sqs, err := messagequeue.NewSQSQueue(c, resource)
	if err != nil {
		log.Warn("queue NewQueue: Unable to create new SQS Queue: %v", err)
		return nil, errors.New(fmt.Sprintf("queue NewQueue: Unable to create new SQS Queue: %v", err))
	}
	//buffparams := messagequeue.BufferParameters{1, 10 * time.Second}
	s := sqs.Sender()
	q := &Queue{Q: sqs, S: &s}
	return q, nil
}

// grab a job from the queue to work on by a free worker
func (q *Queue) Dequeue() (*Job, error) {
	r := *q.R
	messages, err := r.Receive()
	if err != nil {
		return nil, errors.New(fmt.Sprintf("queue Dequeue: Receive message failed: %v", err))
	}
	log.Debug("queue Dequeue: Received messages: %v", messages)
	msg := messages[0]
	data, err := msg.Bytes()
	var j Job
	err = json.Unmarshal(data, &j)
	if err != nil {
		log.Warn("queue Dequeue: Parsing of message failed: %v", err)
		return nil, errors.New(fmt.Sprintf("queue Dequeue: Parsing of message failed: %v", err))
	}

	log.Debug("queue Dequeue: Received job: %v", j)
	return &j, nil

}

// Generate a UUID for the job
func get_uuid() (string, error) {
	uuid := make([]byte, 16)
	nbytes, err := rand.Read(uuid)
	if (err != nil) || (len(uuid) != nbytes) {
		return "", err
	}
	uuid[6] = (uuid[6] & 0x0f) | 0x40 // RFC 4122
	uuid[8] = (uuid[8] & 0x3f) | 0x80 // RFC 4122
	return hex.EncodeToString(uuid), nil
}

func NewJobQueue(db *sqlx.DB) *JobQueue {
	q := &JobQueue{}
	q.DB = db
	return q
}

// create a new orchestration job
func (jq *JobQueue) Add(j *Job) error {
	uuid, _ := get_uuid()
	j.UUID = &uuid
	tx, err := jq.DB.Beginx()
	if err != nil {
		return fmt.Errorf("JobQueue Add: Begin transaction failed: %v", err)
	}
	defer tx.Commit()
	_, err = tx.NamedQuery(`INSERT INTO orchestrations (
                uuid, repository, sha, rollbackstrategy, discovery, deploymentstyle, environment, creator, status
        ) VALUES (
                :uuid, :repository, :sha, :rollbackstrategy, :discovery, :deploymentstyle, :environment, :creator, :status
        ) RETURNING ID`,
		j,
	)
	if err != nil {
		return fmt.Errorf("JobQueue Add: Error adding job to db: %v", err)
	}
	return nil
}

// update a job
func (jq *JobQueue) Update(j *Job) error {
	uuid, _ := get_uuid()
	j.UUID = &uuid
	// TODO not implemented yet
	return nil
}

// delete a job
func (jq *JobQueue) Delete(uuid string) error {
	psql := squirrel.StatementBuilder.PlaceholderFormat(squirrel.Dollar)
	query := psql.Delete("*").From("orchestrations")
	query = query.Where(fmt.Sprintf("%s = ?", "uuid"), uuid)
	query = query.Limit(1)
	q, args, err := query.ToSql()
	if err != nil {
		return fmt.Errorf("JobQueue Delete: Error formulating query: %v", err)
	}
	_, err = jq.DB.Exec(q, args...)
	if err != nil {
		return fmt.Errorf("JobQueue Delete: Error deleting job in db: %v", err)
	}
	return nil
}

// enqueue the job to be worked on some more
func (q *Queue) Enqueue(j *Job) error {
	s := *q.S
	qq := *q.Q
	msgdata, err := json.Marshal(j)
	if err != nil {
		return fmt.Errorf("JobQueue Enqueue: Unable to marshal into JSON: %v", err)
	}
	m, err := qq.Message(msgdata)
	if err != nil {
		return fmt.Errorf("JobQueue Enqueue: Message format failed: %v", err)
	}
	err = s.Send(m)
	if err != nil {
		return fmt.Errorf("JobQueue Enqueue: Unable to send message: %v", err)
	}
	return nil
}

// examine a job whether it is owned by us or not
func (jq *JobQueue) Peek(uuid string) (*Job, error) {
	psql := squirrel.StatementBuilder.PlaceholderFormat(squirrel.Dollar)
	result := Job{}
	query := psql.Select("*").From("orchestrations")
	query = query.Where(fmt.Sprintf("%s = ?", "uuid"), uuid)
	query = query.Limit(1)
	q, args, err := query.ToSql()
	if err != nil {
		return nil, fmt.Errorf("JobQueue Peek: Error formulating query: %v", err)
	}
	err = jq.DB.QueryRowx(q, args...).StructScan(&result)
	if err != nil {
		return nil, fmt.Errorf("JobQueue Peek: Error querying db: %v", err)
	}
	return &result, nil
}

// list all jobs
func (jq *JobQueue) List(options *ListJobOptions) ([]*Job, error) {
	psql := squirrel.StatementBuilder.PlaceholderFormat(squirrel.Dollar)

	results := []*Job{}
	query := psql.Select("*").From("orchestrations")
	for k, v := range map[string]*string{
		"repository":  options.Repository,
		"environment": options.Environment,
	} {
		if v != nil {
			query = query.Where(fmt.Sprintf("%s = ?", k), *v)
		}
	}

	q, args, err := query.ToSql()
	if err != nil {
		return nil, fmt.Errorf("JobQueue List: Error formulating query: %v", err)
	}

	rows, err := jq.DB.Queryx(q, args...)
	if err != nil {
		return nil, fmt.Errorf("JobQueue List: Error querying db: %v", err)
	}
	defer rows.Close()

	for rows.Next() {
		var job Job
		err := rows.StructScan(&job)
		if err != nil {
			return nil, fmt.Errorf("JobQueue List: Error scanning results: %v", err)
		}
		results = append(results, &job)
	}
	return results, nil
}
