package poormanqueue

import (
	"encoding/json"

	"github.com/hashicorp/consul/api"
)

// consulqueue is a poor mans queue built on top of consul.
//
// All write operations are done using check+set.
type ConsulQueue struct {
	key          string
	consulClient *api.Client
}

type workFunc func([]string) ([]string, error)

var queryOptions = &api.QueryOptions{
	RequireConsistent: true,
}

func NewConsulQueue(key string, consulClient *api.Client) (*ConsulQueue, error) {
	return &ConsulQueue{key, consulClient}, nil
}

func (q *ConsulQueue) Enqueue(items ...string) error {
	return q.unshift(items...)
}

func (q *ConsulQueue) Dequeue() (*string, error) {
	return q.pop()
}

func (q *ConsulQueue) Drain() (*string, int, error) {
	return q.drain()
}

// cas will fetch the data from consul. Unmarshal the json. Then it will run
// the worker script. Remarshal the json and then attempt to put it. If it
// fails to do it atomically it will attempt it all over again.
func (q *ConsulQueue) cas(worker workFunc) error {
	for {
		kv, _, err := q.consulClient.KV().Get(q.key, queryOptions)
		if err != nil {
			return err
		}

		data := []string{}

		// If the key exists we can decode it. If not we need to create a KV pair.
		if kv != nil {
			err = json.Unmarshal(kv.Value, &data)
			if err != nil {
				return err
			}
		} else {
			kv = &api.KVPair{}
			kv.Key = q.key
			// If the ModifyIndex is set to 0 then consul will ensure that the
			// key still doesn't exist.
			kv.ModifyIndex = 0
		}

		data, err = worker(data)
		if err != nil {
			return err
		}

		kv.Value, err = json.Marshal(data)
		if err != nil {
			return err
		}

		success, _, err := q.consulClient.KV().CAS(kv, nil)
		if err != nil {
			return err
		}

		if success {
			break
		}
	}

	return nil
}

// Push will append items to the end of the queue.
func (q *ConsulQueue) push(items ...string) error {
	err := q.cas(
		func(data []string) ([]string, error) {
			data = append(data, items...)
			return data, nil
		},
	)
	if err != nil {
		return err
	}

	return nil
}

// Pop will fetch the last item from the queue.
func (q *ConsulQueue) pop() (*string, error) {
	var res *string

	err := q.cas(
		func(data []string) ([]string, error) {
			if len(data) >= 1 {
				res = &(data[len(data)-1])

				return data[:len(data)-1], nil
			} else {
				res = nil
			}

			return data, nil
		},
	)
	if err != nil {
		return nil, err
	}

	return res, nil
}

// Drain will empty the queue, returning the last item and a count of the items
// that were in the queue.
func (q *ConsulQueue) drain() (*string, int, error) {
	var res *string
	var count int

	err := q.cas(
		func(data []string) ([]string, error) {
			count = len(data)

			if count >= 1 {
				res = &(data[len(data)-1])
				return []string{}, nil
			} else {
				res = nil
			}

			return data, nil
		},
	)
	if err != nil {
		return nil, 0, err
	}

	return res, count, nil
}

// UnShift will prepend items to the queue.
func (q *ConsulQueue) unshift(items ...string) error {
	err := q.cas(
		func(data []string) ([]string, error) {
			data = append(items, data...)
			return data, nil
		},
	)
	if err != nil {
		return err
	}

	return nil
}

// Shift will grab the first item from the queue.
func (q *ConsulQueue) shift() (*string, error) {
	var res *string

	err := q.cas(
		func(data []string) ([]string, error) {
			if len(data) >= 1 {
				res = &(data[0])

				return data[1:], nil
			} else {
				res = nil
			}

			return data, nil
		},
	)
	if err != nil {
		return nil, err
	}

	return res, nil
}
