package poormanqueue

import (
	"fmt"
	"path"
	"sync"

	"github.com/hashicorp/consul/api"
)

// QueueManager is a wrapper to help you manage a bunch of queues in a
// namespace.
type QueueManager struct {
	prefix   string
	queues   map[string]Queue
	lock     *sync.RWMutex
	newQueue func(string) (Queue, error)
}

// Create a QueueManager backed by 'ConsulQueue's. Queues will be stored under
// the 'prefix' directory in the consul kv.
func NewConsulQueueManager(prefix string, consulClient *api.Client) (*QueueManager, error) {
	return &QueueManager{
		prefix:   prefix,
		queues:   make(map[string]Queue),
		lock:     &sync.RWMutex{},
		newQueue: func(key string) (Queue, error) { return NewConsulQueue(key, consulClient) },
	}, nil
}

func NewLocalQueueManager() (*QueueManager, error) {
	return &QueueManager{
		prefix:   "",
		queues:   make(map[string]Queue),
		lock:     &sync.RWMutex{},
		newQueue: func(key string) (Queue, error) { return NewLocalQueue(), nil },
	}, nil
}

func (qm *QueueManager) addPrefix(key string) string {
	if qm.prefix != "" {
		return path.Join(qm.prefix, key)
	}

	return key
}

// InitQueue will do all the work to initialize a queue. It will return true
// if a queue was created. False if the queue already exists.
func (qm *QueueManager) InitQueue(key string) (bool, error) {
	qm.lock.Lock()
	defer qm.lock.Unlock()

	if _, ok := qm.queues[key]; !ok {
		queue, err := qm.newQueue(qm.addPrefix(key))
		if err != nil {
			return false, err
		}

		qm.queues[key] = queue
		return true, nil
	}

	return false, nil
}

func (qm *QueueManager) Enqueue(key string, items ...string) error {
	qm.lock.RLock()
	defer qm.lock.RUnlock()

	if queue, ok := qm.queues[key]; ok {
		return queue.Enqueue(items...)
	}

	return fmt.Errorf("Queue %q is not initalized", key)
}

func (qm *QueueManager) Dequeue(key string) (*string, error) {
	qm.lock.RLock()
	defer qm.lock.RUnlock()

	if queue, ok := qm.queues[key]; ok {
		return queue.Dequeue()
	}

	return nil, fmt.Errorf("Queue %q is not initalized", key)
}

func (qm *QueueManager) Drain(key string) (*string, int, error) {
	qm.lock.RLock()
	defer qm.lock.RUnlock()

	if queue, ok := qm.queues[key]; ok {
		return queue.Drain()
	}

	return nil, 0, fmt.Errorf("Queue %q is not initalized", key)
}

func (qm *QueueManager) ListQueues() ([]string, error) {
	return []string{}, nil
}
