module "base" {
    source = "../base"
}

# Should be moved to a base module
provider "aws" {
  region = "us-west-2"
}

########
# godoc #
########

resource "aws_instance" "godoc" {
  count = 1
  # Force an ami version to prevent terraform from rebuilding:
  ami = "${module.base.ami}"
  instance_type = "c4.large"
  subnet_id = "${element(split(",", module.base.prod_avail), count.index % 3)}"
  security_groups = ["${module.base.prod_twitch_security_group}"]
  tags {
    Name = "godoc-${count.index}.prod.us-west2.justin.tv"
  }
  user_data = <<END_OF_STRING
#cloud-config
hostname: "godoc-${count.index}"
fqdn: "godoc-${count.index}.prod.us-west2.justin.tv"
manage_etc_hosts: true
runcmd:
 - sleep 1
 - echo "cluster=godoc" > /etc/facter/facts.d/cluster.txt
 - echo "twitch_environment=production" > /etc/facter/facts.d/twitch_environment.txt
 - echo "app_env=production" > /etc/facter/facts.d/app_env.txt
 - echo "clean=true" > /etc/facter/facts.d/clean.txt
 - puppet agent --test
END_OF_STRING
}

resource "aws_route53_record" "godoc" {
  count = 1
  zone_id = "${module.base.r53}"
  name = "godoc-${count.index}.prod.us-west2.justin.tv"
  type = "A"
  ttl = "300"
  records = ["${element(aws_instance.godoc.*.private_ip, count.index)}"]
} 


#######
# SSL #
#######

resource "aws_route53_record" "godoc-elb" {
  count = 1
  zone_id = "${module.base.r53}"
  name = "godoc.prod.us-west2.justin.tv"
  type = "CNAME"
  ttl = "300"
  records = ["${aws_elb.godoc.dns_name}"]
}

resource "aws_elb" "godoc" {
  name = "godoc"
  subnets = ["${aws_instance.godoc.*.subnet_id}"]
  internal = true

  listener {
    instance_port = 80
    instance_protocol = "http"
    lb_port = 80
    lb_protocol = "http"
  }

  listener {
    instance_port = 80
    instance_protocol = "http"
    lb_port = 443
    lb_protocol = "https"
    ssl_certificate_id = "arn:aws:iam::673385534282:server-certificate/wildcard.internal.justin.tv"
  }

  health_check {
    healthy_threshold = 2
    unhealthy_threshold = 2
    timeout = 3
    target = "HTTP:80/running"
    interval = 30
  }

  security_groups = ["${module.base.prod_twitch_security_group}"]
  instances = ["${aws_instance.godoc.*.id}"]
  cross_zone_load_balancing = true
}

###########
# Outputs #
###########

output "godoc_0_hostname" {
  value = "${aws_route53_record.godoc.name}"
}
