module "base" {
    source = "../base"
}

variable codedeploy_role {
  description = "The name of the codedeploy role to attach to our instance profile"
  default {
    name = "tw-codedeploy-role"
  }
}

# Should be moved to a base module
provider "aws" {
  region = "us-west-2"
}

###############
# Not Brigade #
###############

resource "aws_iam_instance_profile" "not-brigade-instance-profile" {
  name = "not-brigade-instance-profile"
  roles = ["${var.codedeploy_role.name}"]
}

resource "aws_instance" "not-brigade" {
  count = 1
  # Force an ami version to prevent terraform from rebuilding:
  ami = "${module.base.ami}"
  instance_type = "c4.large"
  iam_instance_profile = "${aws_iam_instance_profile.not-brigade-instance-profile.name}"
  subnet_id = "${element(split(",", module.base.prod_avail), count.index % 3)}"
  security_groups = ["${module.base.prod_twitch_security_group}"]
  tags {
    Name = "not-brigade-${count.index}.prod.us-west2.justin.tv"
  }
  user_data = <<END_OF_STRING
#cloud-config
hostname: "not-brigade-${count.index}"
fqdn: "not-brigade-${count.index}.prod.us-west2.justin.tv"
manage_etc_hosts: true
runcmd:
 - sleep 1
 - echo "cluster=not-brigade" > /etc/facter/facts.d/cluster.txt
 - echo "twitch_environment=production" > /etc/facter/facts.d/twitch_environment.txt
 - echo "app_env=production" > /etc/facter/facts.d/app_env.txt
 - echo "clean=true" > /etc/facter/facts.d/clean.txt
 - puppet agent --test --server=puppetmaster-6dbb10.sfo01.justin.tv
END_OF_STRING
}

resource "aws_route53_record" "not-brigade" {
  count = 1
  zone_id = "${module.base.r53}"
  name = "not-brigade-${count.index}.prod.us-west2.justin.tv"
  type = "A"
  ttl = "300"
  records = ["${element(aws_instance.not-brigade.*.private_ip, count.index)}"]
} 

#######
# ELB #
#######

resource "aws_route53_record" "not-brigade-elb" {
  count = 1
  zone_id = "${module.base.r53}"
  name = "not-brigade.prod.us-west2.justin.tv"
  type = "CNAME"
  ttl = "300"
  records = ["${aws_elb.not-brigade.dns_name}"]
}

resource "aws_elb" "not-brigade" {
  name = "not-brigade"
  subnets = ["${aws_instance.not-brigade.*.subnet_id}"]
  internal = true

  #listener {
  #  instance_port = 80
  #  instance_protocol = "http"
  #  lb_port = 80
  #  lb_protocol = "http"
  #}

  listener {
    instance_port = 8045
    instance_protocol = "http"
    lb_port = 443
    lb_protocol = "https"
    ssl_certificate_id = "arn:aws:iam::673385534282:server-certificate/wildcard.internal.justin.tv"
  }

  health_check {
    healthy_threshold = 2
    unhealthy_threshold = 2
    timeout = 3
    target = "HTTP:8045/running"
    interval = 30
  }

  security_groups = ["${module.base.prod_twitch_security_group}"]
  instances = ["${aws_instance.not-brigade.*.id}"]
  cross_zone_load_balancing = true
}

output "not-brigade-url" {
  value = "${aws_route53_record.not-brigade-elb.name}"
}
