variable "hostname_prefix" {}
variable "facts" {}
variable "cluster" {}
variable "instance_type" {
	default = "m3.medium"
}

// Defaults to hardcoded security group because concat() with an empty list results in an empty array element which breaks things :(
variable "security_groups" {
	default = "sg-27d18d42"
}

variable "count" {
	default = 1
}

variable "root_volume_size" {
	default = 8
}

variable "ami" {
  default = "ami-1d1e0f2d"
}

variable "owner" {
	default = "devtools"
}

module "base" {
	source = "git::git+ssh://git@git-aws.internal.justin.tv/release/terraform.git//base"
}

resource "aws_instance" "instance" {
	count = "${var.count}"
	ami = "${var.ami}"
	instance_type = "${var.instance_type}"
	security_groups = ["${concat(module.base.prod_twitch_security_group, split(",", var.security_groups))}"]

	subnet_id = "${element(split(",", module.base.prod_avail), count.index % 3)}"

	user_data = <<END_OF_STRING
#cloud-config
preserve_hostname: true
manage_etc_hosts: false
bootcmd:
 - cloud-init-per instance my_set_hostname sh -xc "echo "${var.hostname_prefix}-$INSTANCE_ID" | sed -e 's/-i-/-/g' > /etc/hostname; hostname -F /etc/hostname"
 - cloud-init-per instance my_etc_hosts sh -xc "sed -i -e '/^127.0.1.1/d' /etc/hosts; echo 127.0.1.1 $(cat /etc/hostname).prod.us-west2.justin.tv $(cat /etc/hostname) >> /etc/hosts"
runcmd:
 - echo "cluster=${var.cluster}" > /etc/facter/facts.d/cluster.txt
 - echo "clean=true" > /etc/facter/facts.d/clean.txt
 - echo "${var.facts}" > /etc/facter/facts.d/terraform.txt
 - puppet agent --test --server=puppetmaster-6dbb10.sfo01.justin.tv
 - puppet agent --test
END_OF_STRING

	tags {
		Owner = "${var.owner}"
		Name = "${var.hostname_prefix}-${var.owner}"
	}
	root_block_device {
		volume_size = "${var.root_volume_size}"
	}
}

resource "aws_route53_record" "instance_dns" {
	count = "${var.count}"
	zone_id = "${module.base.r53}"
	name = "${var.hostname_prefix}-${replace(element(aws_instance.instance.*.id, count.index), "i-", "")}.prod.us-west2.justin.tv"
	type = "A"
	ttl = "300"
	records = ["${element(aws_instance.instance.*.private_ip, count.index)}"]
}

output "fqdn" {
	value = "${join(",", aws_route53_record.instance_dns.*.fqdn)}"
}

output "private_ip" {
	value = "${join(",", aws_instance.instance.*.private_ip)}"
}

output "public_ip" {
	value = "${join(",", aws_instance.instance.*.public_ip)}"
}

output "instance_id" {
	value = "${join(",", aws_instance.instance.*.id)}"
}
