module "base" {
    source = "../base"
}

provider "aws" {
    region = "us-west-2"
}



#################################
# TwitchCon Cms Box             #
#################################

resource "aws_instance" "twitchcon_cms_prod_us_west2_justin_tv" {
  count = 4
  ami = "ami-2f82b11f"
  instance_type = "m3.large"
  iam_instance_profile = "twitchcon-cms"
  subnet_id = "${element(split(",", module.base.prod_avail), count.index % 3)}"
  security_groups = ["${aws_security_group.twitchcon_cms_security_group.id}"]
    user_data = <<END_OF_STRING
#cloud-config
hostname: "twichcon-cms-${count.index}"
fqdn: "twichcon-cms-${count.index}.prod.us-west2.justin.tv"
manage_etc_hosts: true
runcmd:
 - sleep 1
 - echo "cluster=twitchcon-cms" > /etc/facter/facts.d/cluster.txt
 - echo "twitch_environment=production" > /etc/facter/facts.d/twitch_environment.txt
 - echo "app_env=production" > /etc/facter/facts.d/app_env.txt
 - echo "clean=true" > /etc/facter/facts.d/clean.txt
 - puppet agent --test
END_OF_STRING

  tags {
    Name = "twitchcon-cms-${count.index}.prod.us-west2.justin.tv"
    Owner = "swong@twitch.tv"
  }
  root_block_device {
    volume_size = 64
  }
}

resource "aws_route53_record" "twitchcon-cms-servers" {
  count = 4
  zone_id = "${module.base.r53}"
  name = "twitchcon-cms-${count.index}.prod.us-west2.justin.tv"
  type = "A"
  ttl = "300"
  records = ["${element(aws_instance.twitchcon_cms_prod_us_west2_justin_tv.*.private_ip, count.index)}"]
} 


###############
# ELB Private #
###############

resource "aws_elb" "twitchcon-cms" {
  name = "twitchcon-cms"
  subnets = ["${aws_instance.twitchcon_cms_prod_us_west2_justin_tv.*.subnet_id}"]
  internal = true
  security_groups = ["${aws_security_group.twitchcon_cms_security_group.id}"]

  listener {
    instance_port = 8080
    instance_protocol = "http"
    lb_port = 80
    lb_protocol = "http"
  }

  listener {
    instance_port = 8080
    instance_protocol = "http"
    lb_port = 443
    lb_protocol = "https"
    ssl_certificate_id = "arn:aws:iam::673385534282:server-certificate/wildcard.internal.justin.tv"
  }

  health_check {
    healthy_threshold = 2
    unhealthy_threshold = 2
    timeout = 3
    target = "HTTP:8080/running"
    interval = 30
  }

  instances = ["${aws_instance.twitchcon_cms_prod_us_west2_justin_tv.*.id}"]
  cross_zone_load_balancing = true
}

resource "aws_route53_record" "twitchcon-cms-elb" {
  count = 1
  zone_id = "${module.base.r53}"
  name = "twitchcon-cms.prod.us-west2.justin.tv"
  type = "CNAME"
  ttl = "300"
  records = ["${aws_elb.twitchcon-cms.dns_name}"]
}

##############
# ELB Public #
##############
resource "aws_elb" "twitchcon-cms-public" {
  name = "twitchcon-cms-public"
  subnets = ["${aws_instance.twitchcon_cms_prod_us_west2_justin_tv.*.subnet_id}"]
  internal = false
  security_groups = ["${aws_security_group.twitchcon_cms_security_group.id}"]

  listener {
    instance_port = 8080
    instance_protocol = "http"
    lb_port = 80
    lb_protocol = "http"
  }

  listener {
    instance_port = 8080
    instance_protocol = "http"
    lb_port = 443
    lb_protocol = "https"
    ssl_certificate_id = "arn:aws:iam::673385534282:server-certificate/wildcard.internal.justin.tv"
  }

  health_check {
    healthy_threshold = 2
    unhealthy_threshold = 2
    timeout = 3
    target = "HTTP:8080/running"
    interval = 30
  }

  instances = ["${aws_instance.twitchcon_cms_prod_us_west2_justin_tv.*.id}"]
  cross_zone_load_balancing = true
}

resource "aws_route53_record" "twitchcon-cms-public-elb" {
  count = 1
  zone_id = "${module.base.public_twitch_r53}"
  name = "twitchcon-cms.prod.us-west2.twitch.tv"
  type = "CNAME"
  ttl = "300"
  records = ["${aws_elb.twitchcon-cms-public.dns_name}"]
}

##################
# Security Group #
##################

resource "aws_security_group" "twitchcon_cms_security_group"{
  name = "twitchcon-cms-security-group"
  description = "Limit access other boxes that are part of Security Group for twitchcon-cms"
  vpc_id="${module.base.prod_vpc}"

  ingress {
    from_port = 0
    to_port = 0
    protocol = "-1"
    self = true
    cidr_blocks = [
          "10.0.0.0/8",
          "192.16.64.0/21",
          "199.9.248.0/21",
          "185.42.204.0/22",
          "127.0.0.1/32",
          // Twitch Office:
          "38.99.10.97/32",
          "192.168.0.0/16"
          ]
  }
  ingress {
    from_port = 80
    to_port = 80
    protocol = "tcp"
    cidr_blocks = [
        "98.234.86.27/32"
    ]
  }
  ingress {
    from_port = 443
    to_port = 443
    protocol = "tcp"
    cidr_blocks = [
        "98.234.86.27/32"
    ]
  }
    ingress {
    from_port = 22
    to_port = 22
    protocol = "tcp"
    cidr_blocks = [
        "98.234.86.27/32"
    ]
  }
    egress {
    from_port = 0
    to_port = 0
    protocol = "-1"
    self = true
    cidr_blocks = [
        "10.0.0.0/8"
    ]

    }
}


