package main

import (
	"flag"
	"fmt"
	"io"
	"log"
	"os"
	"os/signal"
	"sort"
	"syscall"
	"time"

	"code.justin.tv/common/chitin"
	_ "code.justin.tv/common/golibs/bininfo"
	"code.justin.tv/release/trace/api"
	"golang.org/x/net/context"
	"google.golang.org/grpc"
	"google.golang.org/grpc/codes"
)

var (
	apiHostPort = flag.String("api", "trace-api.prod.us-west2.justin.tv:11143", "Host and port for the Trace API")
	service     = flag.String("svc", "code.justin.tv/web/web", "The service to analyze")
	sampling    = flag.Float64("sample", 0.1, "Sampling rate")
	duration    = flag.Duration("t", time.Second, "How long to capture data before returning a result")
)

func main() {
	flag.Parse()

	err := chitin.ExperimentalTraceProcessOptIn()
	if err != nil {
		log.Fatalf("trace enable err=%q", err)
	}

	conn, err := grpc.Dial(*apiHostPort, grpc.WithInsecure())
	if err != nil {
		log.Fatalf("dial err=%q", err)
	}
	defer conn.Close()
	client := api.NewTraceClient(conn)
	ctx, cancel := context.WithTimeout(context.Background(), *duration)

	sigch := make(chan os.Signal, 1)
	signal.Notify(sigch, syscall.SIGINT)
	go func() {
		<-sigch
		cancel()
	}()

	var comparison api.Comparison
	comparison.ServiceName = &api.StringComparison{Value: *service}

	firehose, err := client.Firehose(ctx, &api.FirehoseRequest{
		Sampling: *sampling,
		Query: &api.Query{
			Comparisons: []*api.Comparison{&comparison},
		},
	})
	if err != nil {
		log.Fatalf("firehose err=%q", err)
	}

	result := consumeFirehose(firehose)
	if _, err := result.WriteTo(os.Stdout); err != nil {
		log.Fatalf("dump err=%q", err)
	}
}

func match(call *api.Call) bool {
	if call.GetSvc().GetName() != *service {
		return false
	}
	return true
}

func consumeFirehose(firehose api.Trace_FirehoseClient) io.WriterTo {
	var tx *api.Transaction
	var err error

	var d = &data{
		routes: make(map[key]*datum),
	}

	for tx, err = firehose.Recv(); err == nil; tx, err = firehose.Recv() {
		var search func(*api.Call)
		search = func(call *api.Call) {
			if match(call) {
				k := key{
					route: call.GetParams().GetHttp().GetRoute(),
				}

				dat, ok := d.routes[k]
				if !ok {
					dat = &datum{}
					d.routes[k] = dat
				}
				dat.count++
				switch code := call.GetParams().GetHttp().GetStatus(); {
				case code >= 200 && code < 300:
					dat.success++
				case code >= 300 && code < 400:
					dat.success++
				}
			}
			for _, sub := range call.Subcalls {
				search(sub)
			}
		}
		search(tx.Root)
	}

	if err != nil {
		switch grpc.Code(err) {
		case codes.DeadlineExceeded:
		default:
			d.err = err
		}
	}
	return d
}

type key struct {
	route string
}

func (k key) String() string {
	return k.route
}

type data struct {
	routes map[key]*datum
	err    error
}

type datum struct {
	count   int
	success int
}

func (d *data) WriteTo(w io.Writer) (int64, error) {
	var nn int64

	const total = "total"
	var longest = len(total)

	var routes []key
	for route := range d.routes {
		routes = append(routes, route)
		if name := route.String(); len(name) > longest {
			longest = len(name)
		}
	}
	sort.Slice(routes, func(i, j int) bool {
		return routes[i].route < routes[j].route
	})

	for _, route := range routes {
		datum := d.routes[route]

		n, err := fmt.Fprintf(w, "%-*s: %5d calls, %6.2f%% success\n",
			longest, route,
			datum.count, 100*float64(datum.success)/float64(datum.count))
		nn += int64(n)
		if err != nil {
			return nn, err
		}
	}

	if d.err != nil {
		return nn, d.err
	}
	return nn, nil
}
