package stats

import "math"

type Histogram struct {
	buckets          []int
	bucketsPerDecade float64
}

func NewHistogram(bucketsPerDecade float64) *Histogram {
	h := &Histogram{
		bucketsPerDecade: bucketsPerDecade,
	}
	h.buckets = make([]int, h.BucketIndex(math.MinInt64)+1)
	return h
}

func (h *Histogram) Add(value int64) {
	h.AddN(value, 1)
}

func (h *Histogram) AddN(value int64, weight int) {
	idx := h.BucketIndex(value)
	h.buckets[idx] += weight
}

func (h *Histogram) BucketIndex(value int64) int {
	if value == 0 {
		return 0
	}
	v := float64(value)
	if value < 0 {
		v = -v
	}

	idx := int(0.5 + h.bucketsPerDecade*(math.Log(v)/math.Log(10)))

	idx = 1 + idx*2
	if value < 0 {
		idx += 1
	}
	return idx
}

func (h *Histogram) BucketBounds(idx int) (int64, int64) {
	if idx == 0 {
		return 0, 1
	}

	sign := 1.0
	if idx%2 == 0 {
		sign = -1.0
	}

	idx = (idx - 1) / 2

	min := math.Exp((float64(idx) - 0.5) * math.Log(10) / h.bucketsPerDecade)
	max := math.Exp((float64(idx) + 0.5) * math.Log(10) / h.bucketsPerDecade)

	if sign == 1 {
		min, max = min+1, max+1
	}
	if sign == -1 {
		min, max = max*sign, min*sign
	}
	return int64(min), int64(max)
}

func (h *Histogram) Iterator() *Iterator {
	it := &Iterator{
		hist: h,
		idx:  -1,
	}
	it.next()
	return it
}

type Iterator struct {
	LowerBound int64
	UpperBound int64
	Weight     int

	hist *Histogram
	idx  int
}

func (it *Iterator) next() bool {
	if it.idx+1 >= len(it.hist.buckets) {
		return false
	}
	it.idx++
	it.LowerBound, it.UpperBound = it.hist.BucketBounds(it.idx)
	it.Weight = it.hist.buckets[it.idx]
	return true
}

func (it *Iterator) NextZigZag() bool {
	return it.next()
}

func (it *Iterator) NextPositive() bool {
	var ok bool
	for l := it.LowerBound; it.LowerBound == l; {
		if it.idx > 0 {
			it.next()
		}
		ok = it.next()
	}
	return ok
}
