package main

import (
	"context"
	"flag"
	"fmt"
	"io"
	"log"
	"net/http"
	"os"

	"code.justin.tv/common/chitin"
	_ "code.justin.tv/common/golibs/bininfo"
	"code.justin.tv/release/trace/internal/deploy"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/codedeploy"
	"github.com/aws/aws-sdk-go/service/lambda"
)

func main() {
	var (
		region = flag.String("region", "us-west-2", "Target AWS region")

		functionName      = flag.String("function-name", "build_index", "Lambda function name")
		aliasName         = flag.String("alias", "current", "Lambda function alias")
		lambdaHandlerName = flag.String("handler-name", "build_index", "Lambda handler name (path within zip file)")
		lambdaHandlerCode = flag.String("handler-file", "", "Path to Lambda handler code (non-zipped)")

		deployAppName     = flag.String("deploy-app-name", "build_index", "CodeDeploy application name")
		deployTargetGroup = flag.String("deploy-target-group", "prod", "CodeDeploy target group name")

		verbose = flag.Bool("v", false, "Enable verbose output")
	)
	flag.Parse()

	err := chitin.ExperimentalTraceProcessOptIn()
	if err != nil {
		log.Fatalf("trace enable err=%q", err)
	}
	rt, _ := chitin.RoundTripper(context.Background(), http.DefaultTransport)

	conf := aws.NewConfig().WithHTTPClient(&http.Client{Transport: rt})
	if *region != "" {
		conf = conf.WithRegion(*region)
	}
	sess := session.New(conf)

	client := &deploy.Client{
		Lambda:     lambda.New(sess),
		CodeDeploy: codedeploy.New(sess),
	}

	if *verbose {
		client.LogfFunc = func(format string, v ...interface{}) {
			log.Printf("info %s", fmt.Sprintf(format, v...))
		}
	}

	if *lambdaHandlerCode == "" {
		log.Fatalf("Lambda handler code must be specified (-handler-file=)")
	}

	ctx := context.Background()

	deployRequest := &deploy.DeployRequest{
		DeployApp:   *deployAppName,
		DeployGroup: *deployTargetGroup,

		LambdaFunction: *functionName,
		LambdaAlias:    *aliasName,
		LambdaFiles: map[string]func() (io.ReadCloser, error){
			*lambdaHandlerName: func() (io.ReadCloser, error) { return os.Open(*lambdaHandlerCode) },
		},
	}

	err = client.Deploy(ctx, deployRequest)
	if err != nil {
		log.Fatalf("deploy failed: %v", err)
	}
}
