package main

import (
	"context"
	"expvar"
	"log"
	"sync/atomic"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials/stscreds"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/cloudwatch"

	"code.justin.tv/common/golibs/pkgpath"
)

func init() {
	pkg, _ := pkgpath.Caller(0)
	expvar.Publish(pkg, expvar.Func(func() interface{} { return exp.dup() }))
}

type expdata struct {
	Accept        int64
	AcceptError   int64
	Scan          int64
	ScanByte      int64
	ScanError     int64
	TCPCloseError int64

	FileOpenError  int64
	FileFlushError int64
	FileCloseError int64
	FileWriteError int64
}

func (ed *expdata) dup() *expdata {
	var out expdata

	out.Accept = atomic.LoadInt64(&ed.Accept)
	out.AcceptError = atomic.LoadInt64(&ed.AcceptError)
	out.Scan = atomic.LoadInt64(&ed.Scan)
	out.ScanByte = atomic.LoadInt64(&ed.ScanByte)
	out.ScanError = atomic.LoadInt64(&ed.ScanError)
	out.TCPCloseError = atomic.LoadInt64(&ed.TCPCloseError)

	out.FileOpenError = atomic.LoadInt64(&ed.FileOpenError)
	out.FileFlushError = atomic.LoadInt64(&ed.FileFlushError)
	out.FileCloseError = atomic.LoadInt64(&ed.FileCloseError)
	out.FileWriteError = atomic.LoadInt64(&ed.FileWriteError)

	return &out
}

func init() {
	go func() {
		ctx := context.Background()

		t := time.NewTimer(15 * time.Second)
		defer t.Stop()

		sess, err := session.NewSession()
		if err != nil {
			log.Printf("session.NewSession err=%q", err)
			return
		}

		creds := stscreds.NewCredentials(sess, dataProducerRoleArn)
		sess, err = session.NewSession(&aws.Config{Credentials: creds, Region: aws.String(region)})
		if err != nil {
			log.Printf("session.NewSession AssumeRole err=%q", err)
			return
		}

		cw := cloudwatch.New(sess)

		var prev, delta expdata
		for range t.C {
			t.Reset(15 * time.Second)

			update := func(cur, prev, delta *int64) {
				val := atomic.LoadInt64(cur)
				d := val - *prev
				*prev, *delta = val, d
			}

			update(&exp.Scan, &prev.Scan, &delta.Scan)
			update(&exp.ScanByte, &prev.ScanByte, &delta.ScanByte)

			now := time.Now()
			ctx, cancel := context.WithTimeout(ctx, 10*time.Second)
			_, err := cw.PutMetricDataWithContext(ctx, &cloudwatch.PutMetricDataInput{
				Namespace: aws.String("twitch-bs-video-trace/pbcollect"),
				MetricData: []*cloudwatch.MetricDatum{
					&cloudwatch.MetricDatum{
						Timestamp:  &now,
						MetricName: aws.String("Events"),
						Unit:       aws.String(cloudwatch.StandardUnitCount),
						Value:      aws.Float64(float64(delta.Scan)),
					},
					&cloudwatch.MetricDatum{
						Timestamp:  &now,
						MetricName: aws.String("EventBytes"),
						Unit:       aws.String(cloudwatch.StandardUnitBytes),
						Value:      aws.Float64(float64(delta.ScanByte)),
					},
				},
			})
			cancel()
			if err != nil {
				log.Printf("cloudwatch.PutMetricData err=%q", err)
				continue
			}
		}
	}()
}
