// Copyright 2009 The Go Authors. All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

// https://golang.org/src/crypto/tls/generate_cert.go

package main

import (
	"crypto/ecdsa"
	"crypto/elliptic"
	"crypto/rand"
	"crypto/tls"
	"crypto/x509"
	"crypto/x509/pkix"
	"encoding/pem"
	"math/big"
	"time"
)

// genCert returns a self-signed TLS certificate.
func genCert() (tls.Certificate, error) {
	var c tls.Certificate

	priv, err := ecdsa.GenerateKey(elliptic.P384(), rand.Reader)
	if err != nil {
		return c, err
	}

	notBefore := time.Unix(0, 0)
	notAfter := notBefore.Add(time.Minute)

	serialNumber, err := rand.Int(rand.Reader, new(big.Int).Lsh(big.NewInt(1), 128))
	if err != nil {
		return c, err
	}

	template := x509.Certificate{
		SerialNumber: serialNumber,
		NotBefore:    notBefore,
		NotAfter:     notAfter,

		Subject: pkix.Name{
			Organization: []string{"go"},
		},
		DNSNames: []string{"go"},

		KeyUsage:              x509.KeyUsageKeyEncipherment | x509.KeyUsageDigitalSignature,
		ExtKeyUsage:           []x509.ExtKeyUsage{x509.ExtKeyUsageServerAuth},
		BasicConstraintsValid: true,
	}

	derBytes, err := x509.CreateCertificate(rand.Reader, &template, &template, &priv.PublicKey, priv)
	if err != nil {
		return c, err
	}

	ecPrivBytes, err := x509.MarshalECPrivateKey(priv)
	if err != nil {
		return c, err
	}

	var (
		certPEMBlock = pem.EncodeToMemory(&pem.Block{Type: "CERTIFICATE", Bytes: derBytes})
		keyPEMBlock  = pem.EncodeToMemory(&pem.Block{Type: "EC PRIVATE KEY", Bytes: ecPrivBytes})
	)

	c, err = tls.X509KeyPair(certPEMBlock, keyPEMBlock)
	if err != nil {
		return c, err
	}

	return c, nil
}
