package main

import (
	"context"
	"fmt"
	"log"
	"net/http"

	thistrace "code.justin.tv/release/trace"
	"code.justin.tv/release/trace/api/report_v1"
	"code.justin.tv/release/trace/persistent"
	"github.com/golang/protobuf/ptypes"
	"github.com/twitchtv/twirp"
	"golang.org/x/net/trace"
)

func handleDebugHTTP(w http.ResponseWriter, r *http.Request) {
	w.Header().Set("Content-Type", "text/plain; charset=utf-8")
	w.Header().Set("X-Content-Type-Options", "nosniff")
	w.WriteHeader(http.StatusOK)

	fmt.Fprintf(w, "proto=%q", r.Proto)
}

func handleDebugRunning(w http.ResponseWriter, r *http.Request) {
}

type reportServer struct {
	index      persistent.ReportIndex
	htmlPrefix string
}

func (srv *reportServer) ServeHTTP(w http.ResponseWriter, r *http.Request) {
	log.Printf("received request for url %q", r.URL.Path)
	w.WriteHeader(http.StatusNotFound)
}

func (srv *reportServer) twirpHooks() *twirp.ServerHooks {
	hooks := new(twirp.ServerHooks)

	spanKey := new(int)

	hooks.RequestRouted = func(ctx context.Context) (context.Context, error) {
		m, mok := twirp.MethodName(ctx)
		s, sok := twirp.ServiceName(ctx)
		if !mok || !sok {
			return ctx, nil
		}

		tr := trace.New("/"+s+"/"+m, m)
		ctx = trace.NewContext(ctx, tr)
		ctx = context.WithValue(ctx, spanKey, tr)

		return ctx, nil
	}

	hooks.Error = func(ctx context.Context, e twirp.Error) context.Context {
		tr, tok := ctx.Value(spanKey).(trace.Trace)
		if tok {
			tr.SetError()
			tr.LazyPrintf("error code=%s message=%q", e.Code(), e.Msg())
		}

		return ctx
	}

	hooks.ResponseSent = func(ctx context.Context) {
		tr, tok := ctx.Value(spanKey).(trace.Trace)
		if tok {
			defer tr.Finish()

			code, cok := twirp.StatusCode(ctx)
			if cok {
				tr.LazyPrintf("status-code=%q", code)
			}
		}
	}

	return hooks
}

func (srv *reportServer) ListReports(ctx context.Context, req *report_v1.ListReportsRequest) (*report_v1.ListReportsResponse, error) {
	var resp report_v1.ListReportsResponse

	reports, err := srv.index.List(ctx)
	if err != nil {
		return nil, err
	}

	for _, rep := range reports {
		mod, err := ptypes.TimestampProto(rep.ModTime())
		if err != nil {
			return nil, err
		}
		resp.Report = append(resp.Report, &report_v1.Report{
			Id:       rep.ID(),
			Modified: mod,
		})
	}

	return &resp, nil
}

func (srv *reportServer) LookupTransaction(ctx context.Context, req *report_v1.LookupTransactionRequest) (*report_v1.LookupTransactionResponse, error) {
	var resp report_v1.LookupTransactionResponse

	reports, err := srv.index.List(ctx)
	if err != nil {
		return nil, err
	}

	var rep persistent.Report
	for _, r := range reports {
		if r.ID() == req.ReportId {
			rep = r
			break
		}
	}
	if rep == nil {
		return nil, fmt.Errorf("report not found")
	}

	txid := thistrace.ParseID(req.TransactionId)
	if txid == nil {
		return nil, fmt.Errorf("invalid txid")
	}

	tx, err := rep.Pluck(ctx, txid)
	if err != nil {
		return nil, err
	}
	resp.Transaction = tx
	return &resp, nil
}

func (srv *reportServer) getReport(ctx context.Context, id string) (persistent.Report, error) {
	reports, err := srv.index.List(ctx)
	if err != nil {
		return nil, err
	}

	for _, r := range reports {
		if r.ID() == id {
			return r, nil
		}
	}
	return nil, fmt.Errorf("report not found")
}

func (srv *reportServer) GetProgramReport(ctx context.Context, req *report_v1.GetProgramReportRequest) (*report_v1.GetProgramReportResponse, error) {
	rep, err := srv.getReport(ctx, req.ReportId)
	if err != nil {
		return nil, err
	}

	content, err := rep.Content(ctx, req.GetProgram().GetName())
	if err != nil {
		return nil, err
	}

	return &report_v1.GetProgramReportResponse{Content: content}, nil
}

func (srv *reportServer) ListProgramsInReport(ctx context.Context, req *report_v1.ListProgramsInReportRequest) (*report_v1.ListProgramsInReportResponse, error) {
	rep, err := srv.getReport(ctx, req.ReportId)
	if err != nil {
		return nil, err
	}

	servers, err := rep.ListServers(ctx)
	if err != nil {
		return nil, err
	}

	var resp report_v1.ListProgramsInReportResponse

	for _, server := range servers {
		resp.Programs = append(resp.Programs, &report_v1.ProgramRef{Name: server})
	}

	return &resp, nil
}
