package main

import (
	"context"
	"io/ioutil"
	"net/http"
	"os"
	"path"
	"reflect"
	"sort"
	"testing"

	"code.justin.tv/release/trace/api"
	"code.justin.tv/release/trace/api/report_v1"
	"code.justin.tv/release/trace/persistent"
)

func TestHandlers(t *testing.T) {

	dir, err := ioutil.TempDir("", "db")
	if err != nil {
		t.Fatalf("TempDir; err = %q", err)
	}
	defer os.RemoveAll(dir)

	for _, tt := range []struct {
		name    string
		svcname string
		txid    [2]uint64
	}{
		{
			name:    "1234",
			svcname: "foo",
			txid:    [2]uint64{0, 1},
		},
		{
			name:    "5678",
			svcname: "bar",
			txid:    [2]uint64{0, 7},
		},
	} {
		db, err := persistent.OpenDB(path.Join(dir, tt.name, "db"))
		if err != nil {
			t.Fatalf("persistent.OpenDB; err = %q", err)
		}

		err = db.WriteTransaction(&api.Transaction{
			TransactionId: tt.txid[:],
			Client: &api.Service{
				Name: tt.svcname,
			},
		})
		if err != nil {
			t.Errorf("WriteTransaction; err = %q", err)
		}
		err = db.CompactAll()
		if err != nil {
			t.Errorf("CompactAll; err = %q", err)
		}
		db.Close()
	}

	index, err := persistent.NewFilesystemReportIndex(http.Dir(dir))
	if err != nil {
		t.Fatalf("persistent.NewFilesystemReportIndex; err = %q", err)
	}

	srv := &reportServer{
		index: index,
	}

	ctx := context.Background()

	{
		resp, err := srv.ListReports(ctx, &report_v1.ListReportsRequest{})
		if err != nil {
			t.Errorf("ListReports; err = %q", err)
		} else if have, want := len(resp.Report), 2; have != want {
			t.Errorf("len(resp.Report); %d != %d", have, want)
		}
		var ids []string
		for _, rep := range resp.Report {
			ids = append(ids, rep.Id)
		}
		sort.Strings(ids)
		if have, want := ids, []string{"1234", "5678"}; !reflect.DeepEqual(have, want) {
			t.Errorf("report IDs; %q != %q", have, want)
		}
	}

	for _, tt := range []struct {
		name    string
		req     *report_v1.LookupTransactionRequest
		wantErr bool
		wantTx  bool
	}{
		{
			name:    "empty request",
			req:     &report_v1.LookupTransactionRequest{},
			wantErr: true,
		},
		{
			name: "invalid report id",
			req: &report_v1.LookupTransactionRequest{
				ReportId: "oops",
			},
			wantErr: true,
		},
		{
			name: "invalid txid",
			req: &report_v1.LookupTransactionRequest{
				ReportId:      "1234",
				TransactionId: "5",
			},
			wantErr: true,
		},
		{
			name: "non-existent txid",
			req: &report_v1.LookupTransactionRequest{
				ReportId:      "1234",
				TransactionId: "55555555555555555555555555555555",
			},
			wantErr: true,
		},
		{
			name: "report 1234",
			req: &report_v1.LookupTransactionRequest{
				ReportId:      "1234",
				TransactionId: "00000000000000000100000000000000",
			},
			wantTx: true,
		},
		{
			name: "report 5678",
			req: &report_v1.LookupTransactionRequest{
				ReportId:      "5678",
				TransactionId: "00000000000000000700000000000000",
			},
			wantTx: true,
		},
	} {
		t.Run(tt.name, func(t *testing.T) {
			resp, err := srv.LookupTransaction(ctx, tt.req)
			if (err != nil) != tt.wantErr {
				t.Errorf("LookupTransaction(%q);\nwantErr = %t, err = %q", tt.req, tt.wantErr, err)
			}
			if err != nil {
				return
			}
			if (resp.Transaction != nil) != tt.wantTx {
				t.Errorf("LookupTransaction(%q);\nwantTx = %t, Transaction = %q", tt.req, tt.wantTx, resp.Transaction)
			}
		})
	}
}
