package main

import (
	"encoding/json"
	"io/ioutil"
	"log"
	"math"
	"net/url"
	"os"
	"path"
	"path/filepath"
	"sort"
	"time"

	"code.justin.tv/release/trace/analysis"
	"code.justin.tv/release/trace/analysis/render"
	"code.justin.tv/release/trace/analysis/tx"
	"code.justin.tv/release/trace/api/report_v1"
	"code.justin.tv/release/trace/internal/htmlreport"
	"code.justin.tv/release/trace/persistent"
	"github.com/golang/protobuf/ptypes"
)

type Report struct {
	// WritePath is the path to the directory to write report data and
	// assets into.
	WritePath string
	// URLPrefix is put in front of all internal URLs to let this
	// report's assets get served properly. This should be relative to
	// the root path of the server which is serving the HTML and asset
	// files that the report generates.
	URLPrefix string

	Services map[string]tx.TransactionSet

	// local database for storing report and sampled transactions
	DB *persistent.DB
}

func NewReport(writePath string, urlPrefix string) *Report {
	rep := &Report{
		WritePath: writePath,
		URLPrefix: urlPrefix,
		Services:  make(map[string]tx.TransactionSet),
	}
	return rep
}

func clean(txs tx.TransactionSet) tx.TransactionSet {
	brokenDuration := 0
	out := tx.NewInMemoryTransactionSet()
	for _, tx := range txs.All() {
		if tx.RootDuration() != time.Duration(0) {
			out.Add(tx)
		} else {
			brokenDuration += 1
		}
	}
	log.Printf("brokenduration=%d", brokenDuration)
	return out
}

// Populate fills a report with data
func (r *Report) Populate(txs tx.TransactionSet) {
	txs = clean(txs)
	r.Services = analysis.SplitByServer(txs)

	svcNames := make([]string, 0, len(r.Services))
	for name, svc := range r.Services {
		if svc.Len() == 0 {
			delete(r.Services, name)
			continue
		}
		svcNames = append(svcNames, name)
	}

	sort.Strings(svcNames)
	for name, svc := range r.Services {
		log.Printf("service=%q requests=%d", name, svc.Len())
	}
}

func (r *Report) URIFor(subpath string) string {
	return path.Join(r.URLPrefix, subpath)
}

func (r *Report) URIForCallRef(ref *report_v1.CallRef) string {
	return r.URIFor("tx.html") + "?" + (url.Values{
		"id": []string{ref.GetTransactionId()},
	}).Encode()
}

func (r *Report) URIForProgramRef(ref *report_v1.ProgramRef) string {
	return r.URIFor(path.Join("bin", ref.GetName()) + ".html")
}

func (r *Report) writeFile(path string, src []byte) error {
	dest := filepath.Join(r.WritePath, path)
	err := os.MkdirAll(filepath.Dir(dest), 0755)
	if err != nil {
		return err
	}
	err = ioutil.WriteFile(dest, src, 0644)
	if err != nil {
		return err
	}
	return nil
}

func (r *Report) writeAsset(path string) error {
	return r.writeFile(path, render.MustAsset(path))
}

func (r *Report) DumpHTML() error {
	var err error
	if err = r.writeStylesheet(); err != nil {
		return err
	}
	if err = r.writeTransactionScript(); err != nil {
		return err
	}
	if err = r.writeIndex(); err != nil {
		return err
	}
	if err = r.writeTransactionBasePage(); err != nil {
		return err
	}
	if err = r.writeServiceList(); err != nil {
		return err
	}
	for name, svc := range r.Services {
		if err = r.writeTransansactionsForService(name, svc); err != nil {
			return err
		}
	}
	return nil
}

func (r *Report) writeStylesheet() error {
	log.Println("writing stylesheet")
	return r.writeAsset("css/style.css")
}

func (r *Report) writeTransactionScript() error {
	log.Println("writing tx.js")
	return r.writeAsset("js/tx.js")
}

func (r *Report) writeIndex() error {
	log.Println("writing index")

	var programs []*report_v1.ProgramRef
	for svc := range r.Services {
		programs = append(programs, &report_v1.ProgramRef{Name: svc})
	}

	buf, err := htmlreport.RenderProgramIndex(r, programs)
	if err != nil {
		return err
	}
	return r.writeFile("index.html", buf)
}

func (r *Report) writeServiceList() error {
	log.Println("writing service list")
	var err error
	for name, data := range r.Services {
		if err = r.writeServiceReport(name, data); err != nil {
			return err
		}
	}
	return nil
}

func summarizeTransaction(t *tx.Transaction) *report_v1.SubtreeSummary {
	if t == nil {
		return nil
	}
	return &report_v1.SubtreeSummary{
		Call: &report_v1.CallSummary{
			Ref: &report_v1.CallRef{
				TransactionId: t.StringID(),
				Path:          nil, // TODO: show perspective of non-root servers
			},
			ServerDuration: ptypes.DurationProto(t.RootDuration()),
		},
		SubtreeDepth: int64(t.Depth()),
		SubtreeSize:  int64(t.Size()),
	}
}

func buildProgramReport(name string, signatures map[string]*tx.SignatureSet, svcData tx.TransactionSet) *report_v1.ProgramReport {
	rep := &report_v1.ProgramReport{
		Program: &report_v1.ProgramRef{Name: name},

		RootServerTimingDistribution: distributionOfTxSet(svcData),
	}

	rep.Noteworthy = &report_v1.NoteworthyTransactions{
		Slowest: summarizeTransaction(analysis.Percentile(svcData, 1.0, tx.ByTime)),
		Deepest: summarizeTransaction(analysis.Percentile(svcData, 1.0, tx.ByDepth)),
		Largest: summarizeTransaction(analysis.Percentile(svcData, 1.0, tx.BySize)),
	}

	var sigs []string
	for sig := range signatures {
		sigs = append(sigs, sig)
	}
	sort.Strings(sigs)
	for _, key := range sigs {
		sig := signatures[key]
		var summary report_v1.CallSignatureSummary
		for _, eg := range sig.Examples {
			summary.Examples = append(summary.Examples, &report_v1.CallRef{
				TransactionId: eg.StringID(),
				Path:          nil, // TODO: show perspective of non-root servers
			})
		}
		for _, dep := range sig.Dependencies() {
			summary.Dependencies = append(summary.Dependencies,
				&report_v1.ProgramRef{
					Name: dep,
				})
		}
		for _, dep := range sig.AllDependencies() {
			summary.TransitiveDependencies = append(summary.TransitiveDependencies,
				&report_v1.ProgramRef{
					Name: dep,
				})
		}
		summary.Method = &report_v1.MethodRef{
			Program:    rep.Program,
			MethodName: sig.RootRPC(),
		}
		summary.ServerTimingDistribution = distributionOfTxSet(sig)
		rep.Signature = append(rep.Signature, &summary)
	}

	return rep
}

func distributionOfTxSet(set tx.TransactionSet) *report_v1.Distribution {
	var data []float64
	for _, t := range set.All() {
		// TODO: show perspective of non-root servers
		data = append(data, t.RootDuration().Seconds())
	}

	// TODO: pick a growth factor that is losslessly expressed as a float64
	const bucketsPerDecade = 20
	opts := &report_v1.Distribution_BucketOptions_Exponential{
		NumFiniteBuckets: bucketsPerDecade * (3 + 3 + 2), // 1µs to 100s
		GrowthFactor:     math.Pow(10, 1.0/bucketsPerDecade),
		Scale:            time.Microsecond.Seconds(),
	}

	return distributionOfFloat64s(data, opts)
}

func distributionOfFloat64s(data []float64, opts *report_v1.Distribution_BucketOptions_Exponential) *report_v1.Distribution {
	dist := &report_v1.Distribution{
		BucketOptions: &report_v1.Distribution_BucketOptions{
			Options: &report_v1.Distribution_BucketOptions_ExponentialBuckets{
				ExponentialBuckets: opts,
			},
		},
		BucketCounts: make([]int64, opts.NumFiniteBuckets+2),
	}

	dist.Count = int64(len(data))
	if len(data) > 0 {
		dist.Range = &report_v1.Distribution_Range{Min: data[0], Max: data[0]}
	}

	bounds := make([]float64, opts.NumFiniteBuckets+1)
	for i := range bounds {
		bounds[i] = opts.Scale * math.Pow(opts.GrowthFactor, float64(i))
	}

	var sum float64
	for _, val := range data {
		sum += val

		if val < dist.Range.Min {
			dist.Range.Min = val
		}
		if val > dist.Range.Max {
			dist.Range.Max = val
		}

		dist.BucketCounts[sort.Search(len(bounds), func(i int) bool { return val < bounds[i] })]++
	}

	if dist.Count != 0 {
		dist.Mean = sum / float64(dist.Count)
	}

	// trim trailing zeros, per Distribution proto definition
	for i := len(dist.BucketCounts) - 1; i >= 0; i-- {
		if dist.BucketCounts[i] != 0 {
			break
		}
		dist.BucketCounts = dist.BucketCounts[:i]

	}

	return dist
}

func (r *Report) writeServiceReport(name string, svcData tx.TransactionSet) error {
	log.Printf("writing service report for %s\n", name)

	signatures := analysis.SplitBySignature(svcData)

	log.Printf("writing signature transaction reports for %s\n", name)
	for _, sigData := range signatures {
		for _, t := range sigData.Examples {
			if err := r.writeTransactionData(t); err != nil {
				return err
			}
		}
	}

	rep := buildProgramReport(name, signatures, svcData)

	if r.DB != nil {
		err := r.DB.WriteProgramReport(rep)
		if err != nil {
			return err
		}
	}

	dest := filepath.Join("bin", rep.GetProgram().GetName()+".html")
	home := path.Clean(r.URLPrefix) + "/"
	buf, err := htmlreport.RenderProgramReport(r, home, rep)
	if err != nil {
		return err
	}
	return r.writeFile(dest, buf)
}

func (r *Report) writeTransansactionsForService(name string, data tx.TransactionSet) error {
	if err := r.writeTransactionData(analysis.Percentile(data, 1.0, tx.ByTime)); err != nil {
		return err
	}
	if err := r.writeTransactionData(analysis.Percentile(data, 0.5, tx.ByTime)); err != nil {
		return err
	}
	if err := r.writeTransactionData(analysis.Percentile(data, 0.0, tx.ByTime)); err != nil {
		return err
	}
	if err := r.writeTransactionData(analysis.Percentile(data, 1.0, tx.ByDepth)); err != nil {
		return err
	}
	if err := r.writeTransactionData(analysis.Percentile(data, 1.0, tx.BySize)); err != nil {
		return err
	}
	return nil
}

func (r *Report) writeTransactionData(t *tx.Transaction) error {
	log.Printf("writing data for transaction %s\n", t.StringID())

	dest := filepath.Join("tx", t.StringID()+".json")
	bytes, err := json.Marshal(t.Root)
	if err != nil {
		return err
	}

	if r.DB != nil && t.Tx != nil {
		err := r.DB.WriteTransaction(cleanedAPITransaction(t.Tx))
		if err != nil {
			return err
		}
	}

	return r.writeFile(dest, bytes)
}

func (r *Report) writeTransactionBasePage() error {
	log.Println("writing base page for transaction reports")

	dest := "tx.html"
	buf, err := htmlreport.RenderTransactionBasePage(r)
	if err != nil {
		return err
	}
	return r.writeFile(dest, buf)
}
