package deploy

import "encoding/json"

// An appspec value holds configuration data for CodeDeploy.
//
// It can be serialized to the JSON format described at the link below.
//
// https://docs.aws.amazon.com/codedeploy/latest/userguide/reference-appspec-file-structure.html
type appspec struct {
	Version   json.Number                   `json:"version"`
	Resources []map[string]*appspecResource `json:"Resources"`
	Hooks     []*appspecHook                `json:"Hooks"`
}

type appspecResource struct {
	Type       string            `json:"Type"`
	Properties map[string]string `json:"Properties"`
}

type appspecHook struct {
}

func buildAppspec(req *DeployRequest, currentVersion, targetVersion string) ([]byte, error) {
	spec := &appspec{
		Version: "0.0",
		Resources: []map[string]*appspecResource{
			map[string]*appspecResource{
				req.LambdaFunction: &appspecResource{
					Type: "AWS::Lambda::Function",
					Properties: map[string]string{
						"Name":           req.LambdaFunction,
						"Alias":          req.LambdaAlias,
						"CurrentVersion": currentVersion,
						"TargetVersion":  targetVersion,
					},
				},
			},
		},
		Hooks: []*appspecHook{},
	}

	specBuf, err := json.Marshal(spec)
	if err != nil {
		return nil, err
	}
	return specBuf, nil
}
