package deploy

import (
	"context"
	"io"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/codedeploy"
	"github.com/aws/aws-sdk-go/service/codedeploy/codedeployiface"
	"github.com/aws/aws-sdk-go/service/lambda"
	"github.com/aws/aws-sdk-go/service/lambda/lambdaiface"
)

type Client struct {
	Lambda     lambdaiface.LambdaAPI
	CodeDeploy codedeployiface.CodeDeployAPI
	LogfFunc   func(format string, v ...interface{})
}

type DeployRequest struct {
	DeployApp   string
	DeployGroup string

	LambdaFunction string
	LambdaAlias    string

	// LambdaFiles specifies the files to be included in the ZIP archive
	// uploaded to Lambda, mapping the name of the file within the archive to
	// a function that can be called to obtain an io.ReadCloser for the file's
	// contents. The caller is responsible for closing the reader.
	LambdaFiles map[string]func() (io.ReadCloser, error)
}

func (c *Client) logf(format string, v ...interface{}) {
	if c.LogfFunc != nil {
		c.LogfFunc(format, v...)
	}
}

func (c *Client) Deploy(ctx context.Context, req *DeployRequest) error {
	c.logf("begin-deploy")

	resp, err := c.startDeploy(ctx, req)
	if err != nil {
		return err
	}

	id := aws.StringValue(resp.DeploymentId)

	err = c.CodeDeploy.WaitUntilDeploymentSuccessfulWithContext(ctx, &codedeploy.GetDeploymentInput{
		DeploymentId: aws.String(id),
	})
	if err != nil {
		return err
	}
	c.logf("end-deploy")

	return nil
}

func (c *Client) startDeploy(ctx context.Context, req *DeployRequest) (*codedeploy.CreateDeploymentOutput, error) {
	currentVersion, err := c.currentVersion(ctx, req)
	if err != nil {
		return nil, err
	}

	targetVersion, err := c.uploadCode(ctx, req)
	if err != nil {
		return nil, err
	}

	specBuf, err := buildAppspec(req, currentVersion, targetVersion)
	if err != nil {
		return nil, err
	}

	deployReq := &codedeploy.CreateDeploymentInput{
		ApplicationName:     aws.String(req.DeployApp),
		DeploymentGroupName: aws.String(req.DeployGroup),
		Revision: &codedeploy.RevisionLocation{
			RevisionType: aws.String("String"),
			String_: &codedeploy.RawString{
				Content: aws.String(string(specBuf)),
			},
		},
	}

	deployResp, err := c.CodeDeploy.CreateDeploymentWithContext(ctx, deployReq)
	if err != nil {
		return nil, err
	}
	c.logf("deploy-id=%q", aws.StringValue(deployResp.DeploymentId))
	return deployResp, nil
}

func (c *Client) currentVersion(ctx context.Context, req *DeployRequest) (string, error) {
	aliasResp, err := c.Lambda.GetAliasWithContext(ctx, &lambda.GetAliasInput{
		FunctionName: aws.String(req.LambdaFunction),
		Name:         aws.String(req.LambdaAlias),
	})
	if err != nil {
		return "", err
	}
	c.logf("current-version=%q", aws.StringValue(aliasResp.FunctionVersion))
	return aws.StringValue(aliasResp.FunctionVersion), nil
}

func (c *Client) uploadCode(ctx context.Context, req *DeployRequest) (string, error) {
	zipFile, err := buildZip(req)
	if err != nil {
		return "", nil
	}

	updateResp, err := c.Lambda.UpdateFunctionCodeWithContext(ctx, &lambda.UpdateFunctionCodeInput{
		FunctionName: aws.String(req.LambdaFunction),
		ZipFile:      zipFile,
		Publish:      aws.Bool(true),
	})
	if err != nil {
		return "", err
	}
	c.logf("new-version=%q", aws.StringValue(updateResp.Version))
	return aws.StringValue(updateResp.Version), nil
}
