package deploy

import (
	"archive/zip"
	"bytes"
	"io"
	"os"
	"sort"
)

func buildZip(req *DeployRequest) ([]byte, error) {
	zipFile := new(bytes.Buffer)
	zw := zip.NewWriter(zipFile)
	add := func(name string, fn func() (io.ReadCloser, error)) error {
		rc, err := fn()
		if err != nil {
			return err
		}
		defer rc.Close()

		// We're uploading code: default to marking the files as executable.
		//
		// See https://github.com/aws/aws-lambda-go/issues/13
		header := &zip.FileHeader{Name: name, Method: zip.Deflate}
		header.SetMode(0777 & os.ModePerm)
		w, err := zw.CreateHeader(header)
		if err != nil {
			return err
		}

		_, err = io.Copy(w, rc)
		return err
	}

	var files []string
	for name := range req.LambdaFiles {
		files = append(files, name)
	}
	sort.Strings(files)
	for _, name := range files {
		err := add(name, req.LambdaFiles[name])
		if err != nil {
			return nil, err
		}
	}
	err := zw.Close()
	if err != nil {
		return nil, err
	}

	return zipFile.Bytes(), nil
}
