package easyprof

import (
	"context"
	"fmt"
	"log"

	"code.justin.tv/video/autoprof/profs3"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials/stscreds"
	"github.com/aws/aws-sdk-go/aws/ec2metadata"
	"github.com/aws/aws-sdk-go/aws/endpoints"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/s3"
)

func Go(ctx context.Context) {
	const (
		defaultRegion = endpoints.UsWest2RegionID
		bucket        = "twitch-bs-video-trace-profile"
		roleARN       = "arn:aws:iam::297560788757:role/trace-data-producer"
	)

	go func() {
		sess, err := getSession(ctx, defaultRegion, roleARN)
		if err != nil {
			log.Printf("getSession err=%q", err)
			return
		}

		c := &profs3.Collector{
			S3:       s3.New(sess),
			S3Bucket: bucket,
			OnError: func(err error) error {
				log.Printf("profs3.Collector.Run err=%q", err)
				return nil
			},
		}
		err = c.Run(ctx)
		if err != nil {
			log.Printf("profs3.Collector.Run fatal err=%q", err)
		}
	}()
}

func getSession(ctx context.Context, defaultRegion, roleARN string) (*session.Session, error) {
	config := &aws.Config{}
	if defaultRegion != "" {
		config = config.WithRegion(defaultRegion)
	}

	sess, err := session.NewSession(config)
	if err != nil {
		return nil, fmt.Errorf("session.NewSession: %s", err)
	}

	region, err := ec2metadata.New(sess).Region()
	if err == nil && region != "" {
		config = config.WithRegion(region)
	}

	if roleARN != "" {
		config = config.WithCredentials(stscreds.NewCredentials(sess, roleARN))
		sess, err = session.NewSession(config)
		if err != nil {
			return nil, fmt.Errorf("session.NewSession AssumeRole: %s", err)
		}
	}

	return sess, nil
}
