package index

import (
	"context"
	"errors"
	"fmt"
	"net/url"
	"strings"

	"github.com/syndtr/goleveldb/leveldb"
	"github.com/syndtr/goleveldb/leveldb/util"
)

type indexEntry struct {
	index *indexFile
	data  *kinesisLocation
}

type indexFile struct {
	bucket string
	key    string
}

type kinesisLocation struct {
	txid   string
	shard  string
	seqNum string
}

func (loc *kinesisLocation) leveldbKey() string {
	return fmt.Sprintf("/tx_location/%s/kinesis/%s/%s",
		url.PathEscape(loc.txid),
		url.PathEscape(loc.shard),
		url.PathEscape(loc.seqNum),
	)
}

var errInvalidLevelDBKey = errors.New("invalid LevelDB key")

func fromLeveldbKey(key string) (*kinesisLocation, error) {
	parts := strings.SplitN(key, "/", 7)
	if len(parts) != 6 {
		return nil, errInvalidLevelDBKey
	}
	if parts[0] != "" || parts[1] != "tx_location" || parts[3] != "kinesis" {
		return nil, errInvalidLevelDBKey
	}
	for _, i := range []int{2, 4, 5} {
		var err error
		parts[i], err = url.PathUnescape(parts[i])
		if err != nil {
			return nil, err
		}
	}
	return &kinesisLocation{txid: parts[2], shard: parts[4], seqNum: parts[5]}, nil
}

// A dbLoader runs the provided fn with a read-only instance of the LevelDB
// database located at index.
type dbLoader func(ctx context.Context, index *indexFile, fn func(db *leveldb.DB) error) error

// findInDB finds Transaction entries with a hex-formatted txidPrefix in a
// LevelDB database. It ignores entries up to and including the previousEnd
// paging token.
func findInDB(ctx context.Context, output chan<- *kinesisLocation, db *leveldb.DB,
	previousEnd *kinesisLocation, txidPrefix string) error {

	prefix := fmt.Sprintf("/tx_location/%s", url.PathEscape(txidPrefix))

	it := db.NewIterator(util.BytesPrefix([]byte(prefix)), nil)
	defer it.Release()

	if previousEnd != nil {
		it.Seek([]byte(previousEnd.leveldbKey()))
	}

	for it.Next() && ctx.Err() == nil {
		loc, err := fromLeveldbKey(string(it.Key()))
		if err != nil {
			continue
		}
		select {
		case output <- loc:
		case <-ctx.Done():
		}
	}

	return it.Error()
}
