package index

import (
	"context"
	"reflect"
	"testing"

	"github.com/syndtr/goleveldb/leveldb"
	"github.com/syndtr/goleveldb/leveldb/storage"
	"golang.org/x/sync/errgroup"
)

func TestLevelDBKey(t *testing.T) {
	testcase := func(loc *kinesisLocation, key string) func(t *testing.T) {
		return func(t *testing.T) {
			if have, want := loc.leveldbKey(), key; have != want {
				t.Errorf("leveldbKey(%#v);\n%q\n!=\n%q", loc, have, want)
			}
			have, err := fromLeveldbKey(key)
			if want := loc; !reflect.DeepEqual(have, want) {
				t.Errorf("fromLeveldbKey(%q);\n%+v\n!=\n%+v\nerr = %v", key, have, want, err)
			}
		}
	}

	errcase := func(key string) func(t *testing.T) {
		return func(t *testing.T) {
			if have, _ := fromLeveldbKey(key); have != nil {
				t.Errorf("fromLeveldbKey(%q);\n%+v != nil", key, have)
			}
		}
	}

	t.Run("", testcase(&kinesisLocation{txid: "00112233", shard: "shard-0", seqNum: "42"},
		"/tx_location/00112233/kinesis/shard-0/42"))
	t.Run("", testcase(&kinesisLocation{txid: "00112233", shard: "shard/0", seqNum: "42"},
		"/tx_location/00112233/kinesis/shard%2F0/42"))
	t.Run("", testcase(&kinesisLocation{txid: "00112233", shard: "shard 0", seqNum: "42"},
		"/tx_location/00112233/kinesis/shard%200/42"))
	t.Run("", testcase(&kinesisLocation{txid: "00112233", shard: "shard&0", seqNum: "42"},
		"/tx_location/00112233/kinesis/shard&0/42"))
	t.Run("", testcase(&kinesisLocation{txid: "00112233", shard: "", seqNum: "42"},
		"/tx_location/00112233/kinesis//42"))
	t.Run("", testcase(&kinesisLocation{txid: ";", shard: "shard-0", seqNum: "42"},
		"/tx_location/%3B/kinesis/shard-0/42"))

	t.Run("", errcase(""))
	t.Run("", errcase("/invalid"))
	t.Run("", errcase("/tx_location/00112233/invalid/shard-0/42"))
	t.Run("", errcase("/tx_location-00112233/kinesis/shard-0/42"))
	t.Run("", errcase("prefix/tx_location/00112233/kinesis/shard-0/42"))
	t.Run("", errcase("/tx_location/00112233/kinesis/shard-0/42/suffix"))
	t.Run("", errcase("/tx_location/00112233/kinesis/shard-0/42%"))
}

func TestFindInDB(t *testing.T) {
	var (
		loc41 = &kinesisLocation{txid: "00111122", shard: "shard-0", seqNum: "41"}
		loc42 = &kinesisLocation{txid: "00112233", shard: "shard-0", seqNum: "42"}
		loc43 = &kinesisLocation{txid: "00112244", shard: "shard-0", seqNum: "43"}
	)

	withDB := func(t *testing.T, fn func(db *leveldb.DB)) {
		stor := storage.NewMemStorage()
		defer stor.Close()

		db, err := leveldb.Open(stor, nil)
		if err != nil {
			t.Fatalf("leveldb.Open; err = %v", err)
		}
		defer db.Close()

		var batch leveldb.Batch

		batch.Put([]byte("/other-data"), []byte(""))
		batch.Put([]byte("/tx_location/99999999/invalid/shard-9/99"), []byte(""))
		for _, loc := range []*kinesisLocation{loc41, loc42, loc43} {
			batch.Put([]byte(loc.leveldbKey()), []byte(""))
		}

		err = db.Write(&batch, nil)
		if err != nil {
			t.Fatalf("db.Write; err = %v", err)
		}

		err = db.SetReadOnly()
		if err != nil {
			t.Fatalf("db.SetReadOnly; err = %v", err)
		}

		fn(db)
	}

	testcase := func(limit int, previousEnd *kinesisLocation, txidPrefix string,
		want []*kinesisLocation) func(t *testing.T) {

		return func(t *testing.T) {
			withDB(t, func(db *leveldb.DB) {
				ctx := context.Background()
				ctx, cancel := context.WithCancel(ctx)
				defer cancel()

				eg, ctx := errgroup.WithContext(ctx)

				output := make(chan *kinesisLocation)
				eg.Go(func() error {
					defer close(output)
					return findInDB(ctx, output, db, previousEnd, txidPrefix)
				})

				var have []*kinesisLocation
				eg.Go(func() error {
					defer cancel()
					for val := range output {
						if len(have) >= limit {
							break
						}
						have = append(have, val)
					}
					return nil
				})

				err := eg.Wait()
				if err != nil {
					t.Errorf("findInDB; err = %v", err)
				}

				if !reflect.DeepEqual(have, want) {
					t.Errorf("findInDB output;\n%v\n!=\n%v\n", have, want)
				}
			})
		}
	}

	t.Run("", testcase(10, nil, "0011223", []*kinesisLocation{loc42}))
	t.Run("", testcase(10, nil, "001122", []*kinesisLocation{loc42, loc43}))
	t.Run("", testcase(10, nil, "", []*kinesisLocation{loc41, loc42, loc43}))
	t.Run("", testcase(2, nil, "", []*kinesisLocation{loc41, loc42}))
	t.Run("", testcase(10, loc41, "", []*kinesisLocation{loc42, loc43}))
}
