package metrics

import (
	"context"
	"time"
)

type message struct {
	datum interface{}
	// Immediately write out buffered messages. The datum field will not be set.
	flush func(err error)
}

type sender struct {
	sleep   func(i int) time.Duration
	buffer  dataSlice
	ch      <-chan *message
	send    func(ctx context.Context, data dataSlice) error
	onError func(err error)
}

type dataSlice interface {
	Len() int
	Cap() int
	Append(v interface{})
	SetLen(i int)
}

type untypedSlice struct{ data []interface{} }

func (s *untypedSlice) Len() int             { return len(s.data) }
func (s *untypedSlice) Cap() int             { return cap(s.data) }
func (s *untypedSlice) Append(v interface{}) { s.data = append(s.data, v) }
func (s *untypedSlice) SetLen(i int)         { s.data = s.data[:i] }

func (s *sender) run(ctx context.Context) error {
	t := time.NewTimer(s.sleep(0))
	defer t.Stop()

	for i := 0; ; i++ {
		s.buffer.SetLen(0)

		select {
		case <-ctx.Done():
			return ctx.Err()
		case msg, ok := <-s.ch:
			if !ok {
				return nil
			}
			if msg.flush != nil {
				// no messages are queued at this point; the flush is a success
				msg.flush(nil)
				continue
			}
			s.buffer.Append(msg.datum)
		}

		flush, err := fillBuffer(ctx, s.buffer, t.C, s.ch)
		if err != nil {
			return err
		}

		t.Reset(s.sleep(i + 1))

		err = s.send(ctx, s.buffer)
		if err != nil {
			s.onError(err)
		}
		if flush != nil {
			flush(err)
		}
	}
}

// fillBuffer receives data messages from ch and places them into the provided
// buf. It returns when the buffer is full, when the Context expires or timer
// triggers, when a flush is requested, or when the data channel is closed.
func fillBuffer(ctx context.Context, buf dataSlice,
	t <-chan time.Time, ch <-chan *message) (flush func(error), err error) {

	for buf.Len() < buf.Cap() {
		select {
		case <-ctx.Done():
			return nil, ctx.Err()
		case <-t:
			return nil, nil
		case msg, ok := <-ch:
			if !ok {
				return nil, nil
			}
			if msg.flush != nil {
				return msg.flush, nil
			}
			buf.Append(msg.datum)
		}
	}

	return nil, nil
}

func flush(ctx context.Context, ch chan<- *message) error {
	errc := make(chan error)
	msg := &message{
		flush: func(err error) {
			select {
			case <-ctx.Done():
			case errc <- err:
			}
		},
	}
	select {
	case <-ctx.Done():
		return ctx.Err()
	case ch <- msg:
		select {
		case <-ctx.Done():
			return ctx.Err()
		case err := <-errc:
			return err
		}
	}
}
