package metrics

import (
	"context"
	"fmt"
	"reflect"
	"sync"
	"testing"
	"time"
)

func TestBatchSender(t *testing.T) {
	t.Run("queue and flush mechanics", func(t *testing.T) {
		ctx := context.Background()
		ctx, cancel := context.WithTimeout(ctx, 10*time.Millisecond)
		defer cancel()

		sendErr := fmt.Errorf("send error")
		var (
			opsMu sync.Mutex
			ops   []interface{}
		)

		ch := make(chan *message, 10)

		s := &sender{
			sleep:  func(i int) time.Duration { return 1 * time.Millisecond },
			buffer: &untypedSlice{data: make([]interface{}, 10)},
			ch:     ch,
			send: func(ctx context.Context, ds dataSlice) error {
				opsMu.Lock()
				defer opsMu.Unlock()

				d := ds.(*untypedSlice).data
				// the data array is preallocated and reused; we need to make a copy to keep
				d = append([]interface{}(nil), d...)
				ops = append(ops, d)

				if len(ops) == 1 {
					return sendErr
				}
				return nil
			},
			onError: func(err error) {
				if err != nil {
					if err != sendErr {
						t.Errorf("onError; err = %v", err)
					}
				}
				opsMu.Lock()
				defer opsMu.Unlock()
				ops = append(ops, err)
			},
		}

		// queue items
		ch <- &message{datum: 1}
		ch <- &message{datum: 2}
		ch <- &message{datum: 3}
		ch <- &message{datum: 4}
		// force queued items to be sent .. first send will be an error
		ch <- &message{flush: func(err error) {}}
		// add another item
		ch <- &message{datum: 5}
		// force a small send .. no error this time
		ch <- &message{flush: func(err error) {}}
		// end the test
		ch <- &message{flush: func(err error) { cancel() }}

		err := s.run(ctx)
		if err != context.Canceled {
			t.Errorf("s.run; err = %v", err)
		}

		opsMu.Lock()

		wantOps := []interface{}{
			[]interface{}{1, 2, 3, 4},
			sendErr,
			[]interface{}{5},
		}
		if !reflect.DeepEqual(ops, wantOps) {
			t.Errorf("ops;\n%q\n!=\n%q\n", ops, wantOps)
		}
	})

	t.Run("periodic send calls", func(t *testing.T) {
		ctx := context.Background()
		ctx, cancel := context.WithTimeout(ctx, 10*time.Millisecond)
		defer cancel()

		ch := make(chan *message, 10)

		s := &sender{
			sleep:  func(i int) time.Duration { return 1 * time.Millisecond },
			buffer: &untypedSlice{data: make([]interface{}, 10)},
			ch:     ch,
			send: func(ctx context.Context, ds dataSlice) error {

				cancel() // expect send to be called periodically
				return nil
			},
			onError: func(err error) {},
		}

		ch <- &message{datum: 1}

		err := s.run(ctx)
		if err != context.Canceled {
			// send was not called
			t.Errorf("s.run; err = %v", err)
		}
	})

	t.Run("omit useless send calls", func(t *testing.T) {
		ctx := context.Background()
		ctx, cancel := context.WithTimeout(ctx, 10*time.Millisecond)
		defer cancel()

		ch := make(chan *message, 10)

		s := &sender{
			sleep:  func(i int) time.Duration { return 1 * time.Millisecond },
			buffer: &untypedSlice{data: make([]interface{}, 10)},
			ch:     ch,
			send: func(ctx context.Context, ds dataSlice) error {
				t.Errorf("send called without messages")
				cancel()
				return nil
			},
			onError: func(err error) {},
		}

		err := s.run(ctx)
		if err != context.DeadlineExceeded {
			// send was called
			t.Errorf("s.run; err = %v", err)
		}
	})

	t.Run("do not allocate memory", func(t *testing.T) {
		ctx := context.Background()

		messageCount := 0

		// preallocate this, since it'll be done by the user
		msg := &message{datum: 1}

		fn := func() {
			ctx, cancel := context.WithTimeout(ctx, 10*time.Millisecond)
			defer cancel()

			// force channel allocation, rather than racing against the run
			// method
			_ = ctx.Done()

			ch := make(chan *message, 10)

			s := &sender{
				sleep:  func(i int) time.Duration { return 1 * time.Millisecond },
				buffer: &untypedSlice{data: make([]interface{}, 10)},
				ch:     ch,
				send: func(ctx context.Context, ds dataSlice) error {
					return nil
				},
				onError: func(err error) {},
			}

			var wg sync.WaitGroup

			wg.Add(1)
			go func() {
				defer wg.Done()

				s.run(ctx)
			}()

			for i := 0; i < messageCount; i++ {
				select {
				case <-ctx.Done():
				case ch <- msg:
				}
			}
			cancel()

			wg.Wait()
		}

		var baseline, allocs float64

		messageCount = 0
		baseline = testing.AllocsPerRun(100, fn)

		messageCount = 100
		allocs = testing.AllocsPerRun(100, fn)

		if have, want := allocs-baseline, float64(0); have != want {
			t.Errorf("sending messages resulted in %.0f != %.0f allocations", have, want)
		}
	})
}
